"use client"
import { useEffect, useRef } from "react"
import gsap from "gsap"

export const LootahLogoAnimation = ({ className }: { className?: string }) => {
    const svgRef = useRef<SVGSVGElement | null>(null)
    const circleRef = useRef<SVGCircleElement | null>(null)
    const iconPathsRef = useRef<(SVGPathElement | null)[]>([])
    const textPathsRef = useRef<(SVGPathElement | null)[]>([])

    useEffect(() => {
        if (!svgRef.current) return

        const ctx = gsap.context(() => {
            const circle = circleRef.current
            const iconPaths = iconPathsRef.current.filter(Boolean) as SVGPathElement[]
            const textPaths = textPathsRef.current.filter(Boolean) as SVGPathElement[]

            // Initial state - everything hidden
            gsap.set(svgRef.current, { opacity: 1 })

            // Circle setup - stroke draw animation
            if (circle) {
                const circleLength = circle.getTotalLength()
                gsap.set(circle, {
                    strokeDasharray: circleLength,
                    strokeDashoffset: circleLength,
                    opacity: 1,
                    scale: 0.8,
                    transformOrigin: "102.5px 102.5px"
                })
            }

            // Icon paths setup - hidden and scaled down
            iconPaths.forEach((path) => {
                const length = path.getTotalLength()
                gsap.set(path, {
                    strokeDasharray: length,
                    strokeDashoffset: length,
                    fill: "transparent",
                    stroke: "white",
                    strokeWidth: 1.5,
                    opacity: 0,
                    scale: 0.5,
                    transformOrigin: "102.5px 102.5px"
                })
            })

            // Text paths setup - hidden and offset
            textPaths.forEach((path, i) => {
                gsap.set(path, {
                    opacity: 0,
                    y: 20,
                    x: -10,
                    scale: 0.9,
                })
            })

            // Master timeline
            const tl = gsap.timeline({
                defaults: { ease: "power3.out" },
                delay: 0.2
            })

            // 1. Circle draws and scales up with rotation
            if (circle) {
                tl.to(circle, {
                    strokeDashoffset: 0,
                    scale: 1,
                    rotation: 360,
                    duration: 1.2,
                    ease: "power2.inOut"
                })
            }

            // 2. Icon paths fade in, scale up, and draw simultaneously
            tl.to(iconPaths, {
                opacity: 1,
                scale: 1,
                duration: 0.4,
                stagger: 0.1,
                ease: "back.out(1.7)"
            }, "-=0.6")

            tl.to(iconPaths, {
                strokeDashoffset: 0,
                duration: 0.8,
                stagger: 0.05,
                ease: "power2.out"
            }, "-=0.3")

            // 3. Fill icon paths with a sweep effect
            tl.to(iconPaths, {
                fill: "white",
                duration: 0.5,
                stagger: 0.05,
                ease: "power1.inOut"
            }, "-=0.2")

            // 4. Text reveals with stagger - wave effect
            tl.to(textPaths, {
                opacity: 1,
                y: 0,
                x: 0,
                scale: 1,
                duration: 0.6,
                stagger: {
                    each: 0.04,
                    from: "start"
                },
                ease: "back.out(1.4)"
            }, "-=0.4")

            // 5. Subtle pulse on the whole logo
            tl.fromTo(svgRef.current,
                { scale: 1 },
                {
                    scale: 1.02,
                    duration: 0.15,
                    yoyo: true,
                    repeat: 1,
                    ease: "power1.inOut"
                }
            )

            // 6. Add glow effect to circle
            tl.to(circle, {
                filter: "drop-shadow(0 0 12px rgba(255,255,255,0.6))",
                duration: 0.4,
            }, "-=0.2")

            // Continuous subtle animations after initial reveal

            // Breathing glow on the circle
            gsap.to(circle, {
                filter: "drop-shadow(0 0 20px rgba(59,130,246,0.5))",
                repeat: -1,
                yoyo: true,
                duration: 2,
                ease: "sine.inOut",
                delay: tl.duration()
            })

            // Subtle float animation on the entire logo
            gsap.to(svgRef.current, {
                y: -5,
                repeat: -1,
                yoyo: true,
                duration: 3,
                ease: "sine.inOut",
                delay: tl.duration()
            })

        }, svgRef)

        return () => ctx.revert()
    }, [])

    return (
        <svg
            ref={svgRef}
            className={className || "h-[120px] 3xl:h-[250px]"}
            width="683"
            height="205"
            viewBox="0 0 683 205"
            xmlns="http://www.w3.org/2000/svg"
            style={{ opacity: 0 }}
        >
            {/* Circle */}
            <circle
                ref={circleRef}
                cx="102.5"
                cy="102.5"
                r="102"
                stroke="white"
                fill="none"
                strokeWidth="2"
            />

            {/* Icon paths inside circle */}
            <path
                ref={el => { iconPathsRef.current[0] = el }}
                d="M95.5137 119.983C87.5137 127.483 80.5136 134.183 73.4136 140.883C73.7136 141.383 73.9136 141.783 74.2136 142.283C75.8136 142.383 77.4136 142.583 78.9136 142.583C103.214 142.583 127.513 142.483 151.813 142.683C156.413 142.683 157.413 141.183 157.313 136.983C157.013 127.483 157.214 117.983 157.214 108.483C157.214 106.683 157.014 104.883 156.914 102.683C148.814 102.683 141.114 102.683 133.514 102.683C127.314 102.683 121.214 102.783 115.014 102.583C113.214 102.483 111.514 101.383 109.714 100.783C110.514 98.9829 110.914 96.6829 112.214 95.3829C122.414 84.8829 132.814 74.5828 143.214 64.1828C146.314 61.0828 155.814 59.1829 159.414 61.2829C160.714 61.9829 161.814 64.1828 161.914 65.6828C162.214 71.7828 162.014 77.9828 162.014 84.1828C162.014 87.4828 160.214 89.0828 157.014 89.1828C152.214 89.2828 147.414 89.1828 141.514 89.1828C146.014 83.9828 151.414 84.1829 157.114 83.3829C157.114 78.3829 157.314 73.4829 157.014 68.4829C156.814 65.3829 150.714 64.6829 147.614 67.5829C143.614 71.2829 139.814 75.2828 135.914 79.1828C130.214 84.7828 124.614 90.2829 117.914 96.8829C120.914 97.0829 122.414 97.2829 124.014 97.2829C134.514 97.2829 145.014 97.3829 155.514 97.2829C160.014 97.2829 162.214 98.9829 162.114 103.783C161.914 116.983 162.014 130.083 162.114 143.283C162.114 146.983 160.714 148.183 157.114 148.183C128.014 148.083 98.8136 148.183 69.7136 148.083C68.0136 148.083 66.2137 146.883 64.5137 146.183C65.2137 144.483 65.6135 142.383 66.8135 141.083C71.9135 135.683 77.3136 130.383 82.7136 125.183C85.8136 121.883 89.4137 119.683 95.5137 119.983Z"
                fill="white"
            />
            <path
                ref={el => { iconPathsRef.current[1] = el }}
                d="M93.8135 52.983C97.6135 51.283 99.3137 51.8829 99.5137 56.0829C99.9137 65.4829 100.714 74.983 101.514 85.983C113.614 74.083 124.714 63.083 136.214 51.783C137.514 52.983 138.613 54.0829 139.813 55.1829C132.413 62.6829 125.414 69.9829 118.214 77.1829C96.8136 98.4829 75.4136 119.683 53.9136 140.783C52.4136 142.183 50.0135 142.683 48.1135 143.583C47.5135 141.683 46.4136 139.683 46.4136 137.783C46.3136 120.483 46.3135 103.083 46.3135 85.783C46.3135 78.983 47.7135 77.6829 54.6135 77.6829C58.3135 77.6829 61.9135 77.6829 65.6135 77.6829C65.9135 77.6829 66.2135 77.7829 66.6135 77.8829C64.2135 81.6829 62.6135 82.3829 52.1135 83.6829C52.1135 100.383 52.1135 117.083 52.1135 134.283C53.2135 133.583 54.1136 133.083 54.9136 132.283C68.0136 119.383 81.1136 106.583 93.9136 93.5829C95.5136 91.9829 96.3136 88.8829 96.2136 86.5829C95.5136 75.4829 94.5135 64.183 93.8135 52.983Z"
                fill="white"
            />

            {/* Text paths - LOOTAH TECH */}
            <path
                ref={el => { textPathsRef.current[0] = el }}
                d="M227.414 78.9829C230.414 78.9829 233.213 78.9829 236.813 78.9829C236.813 85.1829 236.813 91.2829 236.813 97.3829C236.813 102.683 236.913 108.083 236.813 113.383C236.813 117.183 238.413 118.883 242.313 118.783C246.613 118.683 251.013 118.783 255.313 118.783C261.813 118.783 261.814 118.783 261.614 125.383C261.614 126.183 261.413 126.983 261.313 128.083C253.313 128.083 245.314 128.583 237.414 127.883C232.414 127.383 227.914 123.683 227.714 118.483C227.014 105.383 227.414 92.2829 227.414 78.9829Z"
                fill="white"
            />
            <path
                ref={el => { textPathsRef.current[1] = el }}
                d="M262.913 102.983C263.413 98.6829 263.213 94.1829 264.613 90.2829C267.913 80.4829 278.613 75.3829 290.413 77.3829C300.413 79.0829 307.313 87.1828 307.513 97.6828C307.613 101.883 307.613 105.983 307.513 110.183C307.013 122.383 298.313 130.183 285.413 130.183C272.413 130.183 263.813 122.383 263.413 110.183C263.313 107.883 263.413 105.483 263.413 103.183C263.213 103.083 263.113 103.083 262.913 102.983ZM297.713 103.583C297.713 101.383 297.813 99.2829 297.613 97.0829C297.213 90.8829 292.613 86.9829 285.813 86.7829C278.513 86.6829 273.813 90.2828 273.313 96.6828C273.013 100.483 273.013 104.283 273.113 108.083C273.313 116.283 277.513 120.383 285.413 120.283C293.313 120.283 297.513 115.983 297.613 107.883C297.813 106.583 297.713 105.083 297.713 103.583Z"
                fill="white"
            />
            <path
                ref={el => { textPathsRef.current[2] = el }}
                d="M311.214 103.683C311.414 99.8831 311.114 95.8831 312.014 92.2831C314.514 81.8831 324.814 75.8831 337.014 77.1831C347.514 78.2831 355.114 86.0831 355.614 96.5831C355.814 101.883 355.914 107.283 355.314 112.583C354.114 124.183 344.814 130.783 331.114 130.083C320.214 129.583 311.914 121.383 311.514 110.683C311.414 108.383 311.514 105.983 311.514 103.683C311.414 103.683 311.314 103.683 311.214 103.683ZM345.714 103.483C345.714 101.783 345.714 100.183 345.714 98.4831C345.614 91.2831 341.514 87.1831 334.314 86.8831C326.414 86.5831 321.614 90.5831 321.214 97.8831C321.014 101.383 321.014 104.883 321.214 108.383C321.514 116.383 325.814 120.583 333.814 120.483C341.714 120.383 345.714 116.183 345.814 107.983C345.714 106.483 345.714 104.983 345.714 103.483Z"
                fill="white"
            />
            <path
                ref={el => { textPathsRef.current[3] = el }}
                d="M373.214 88.3829C367.414 88.3829 362.714 88.3829 357.714 88.3829C357.714 85.0829 357.714 82.2829 357.714 79.1829C371.314 79.1829 384.714 79.1829 398.414 79.1829C398.414 82.1829 398.414 84.9829 398.414 88.4829C393.514 88.4829 388.614 88.4829 383.114 88.4829C383.114 102.083 383.114 114.983 383.114 128.283C379.614 128.283 376.814 128.283 373.214 128.283C373.214 115.083 373.214 102.183 373.214 88.3829Z"
                fill="white"
            />
            <path
                ref={el => { textPathsRef.current[4] = el }}
                d="M441.513 128.283C431.513 129.283 431.513 129.283 431.513 120.383C431.513 117.383 431.513 114.483 431.513 110.583C424.913 110.583 418.613 110.483 412.413 110.783C411.713 110.783 410.613 112.883 410.613 114.083C410.413 118.683 410.513 123.283 410.513 128.283C407.013 128.283 404.213 128.283 400.913 128.283C400.913 116.583 400.113 105.083 401.113 93.783C402.013 83.983 411.513 77.283 421.413 77.483C431.213 77.683 439.313 84.6829 441.113 94.6829C441.413 96.5829 441.513 98.6829 441.513 100.683C441.613 109.783 441.513 118.883 441.513 128.283ZM421.113 100.483C423.313 100.483 425.413 100.383 427.613 100.483C431.213 100.783 431.913 99.083 431.313 95.983C430.413 91.583 426.713 87.983 422.113 87.283C417.913 86.683 413.213 89.483 411.413 93.783C409.313 98.783 410.313 100.383 415.613 100.483C417.413 100.483 419.213 100.483 421.113 100.483Z"
                fill="white"
            />
            <path
                ref={el => { textPathsRef.current[5] = el }}
                d="M447.414 78.9829C450.714 78.9829 453.313 78.9829 456.813 78.9829C456.813 84.5829 456.813 90.0829 456.813 95.8829C464.113 95.8829 470.514 95.8829 477.714 95.8829C477.714 90.4829 477.714 85.0829 477.714 79.2829C481.314 79.2829 484.114 79.2829 487.414 79.2829C487.414 95.4829 487.414 111.683 487.414 127.883C487.014 128.183 486.714 128.483 486.414 128.483C477.914 129.283 478.014 129.283 477.914 120.783C477.814 115.983 479.114 109.783 476.614 106.983C474.414 104.583 467.914 106.083 463.214 105.983C461.414 105.883 459.614 105.983 456.914 105.983C456.914 109.983 456.914 113.583 456.914 117.183C456.914 120.783 456.914 124.483 456.914 128.383C453.514 128.383 450.613 128.383 447.313 128.383C447.413 111.883 447.414 95.5829 447.414 78.9829Z"
                fill="white"
            />
            {/* TECH */}
            <path
                ref={el => { textPathsRef.current[6] = el }}
                d="M543.813 129.883C540.313 129.883 537.713 129.883 534.313 129.883C534.313 125.083 534.313 120.483 534.313 115.883C534.313 108.683 534.213 101.583 534.413 94.3829C534.513 91.1829 533.513 89.6829 530.213 90.0829C528.913 90.1829 527.513 90.0829 526.213 90.0829C517.913 90.0829 517.813 89.983 519.513 80.783C532.513 80.783 545.613 80.783 559.013 80.783C559.013 83.783 559.013 86.6829 559.013 90.1829C554.013 90.1829 549.213 90.1829 543.913 90.1829C543.813 103.683 543.813 116.583 543.813 129.883Z"
                fill="white"
            />
            <path
                ref={el => { textPathsRef.current[7] = el }}
                d="M598.213 90.083C592.313 90.083 586.813 89.8829 581.313 90.1829C574.613 90.4829 571.413 93.583 570.813 100.283C578.713 100.283 586.613 100.283 594.813 100.283C594.813 103.683 594.813 106.383 594.813 109.783C586.813 109.783 578.913 109.783 571.013 109.783C570.713 116.383 574.613 120.183 581.713 120.483C587.113 120.683 592.613 120.483 598.313 120.483C598.313 123.883 598.313 126.683 598.313 130.783C589.913 130.083 581.613 130.083 573.613 128.483C567.013 127.183 562.313 121.983 561.613 115.183C560.813 108.183 560.613 100.783 562.013 93.983C563.713 85.883 570.113 81.483 578.313 80.783C584.713 80.283 591.313 80.6829 598.213 80.6829C598.213 83.5829 598.213 86.483 598.213 90.083Z"
                fill="white"
            />
            <path
                ref={el => { textPathsRef.current[8] = el }}
                d="M637.713 120.483C637.713 123.983 637.713 126.583 637.713 130.583C629.813 130.083 622.013 130.483 614.613 128.783C606.913 127.083 602.313 120.883 601.613 112.783C601.113 107.183 601.213 101.483 601.813 95.883C602.613 88.383 610.013 81.383 617.713 80.783C624.113 80.283 630.613 80.6829 637.513 80.6829C637.513 83.6829 637.513 86.583 637.513 90.083C632.413 90.083 627.513 89.983 622.513 90.083C614.913 90.183 611.213 93.883 610.913 101.383C610.413 114.883 610.413 121.583 629.313 120.483C631.913 120.383 634.613 120.483 637.713 120.483Z"
                fill="white"
            />
            <path
                ref={el => { textPathsRef.current[9] = el }}
                d="M651.714 97.5828C659.014 97.5828 665.314 97.5828 672.214 97.5828C672.214 91.9828 672.214 86.6828 672.214 80.8828C675.614 80.8828 678.314 80.8828 681.714 80.8828C681.814 82.5828 682.014 84.2828 682.014 86.0828C682.014 98.1828 682.014 110.383 682.014 122.483C682.014 130.783 682.014 130.783 672.414 129.583C672.414 123.983 672.214 118.183 672.514 112.483C672.714 108.583 671.414 107.183 667.414 107.383C662.514 107.683 657.513 107.483 651.813 107.483C651.813 111.583 651.813 115.183 651.813 118.783C651.813 122.283 651.813 125.683 651.813 129.683C648.613 129.683 645.714 129.683 642.514 129.683C642.514 113.583 642.514 97.4828 642.514 80.9828C645.414 80.9828 648.214 80.9828 651.714 80.9828C651.714 86.3828 651.714 91.7828 651.714 97.5828Z"
                fill="white"
            />
        </svg>
    )
}
