"use client";

import ReactMarkdown from "react-markdown";
import remarkGfm from "remark-gfm";
import Image from "next/image";
import { getStrapiMediaUrl } from "@/core/lib/api/client";
import { cn } from "@/core/lib/utils";

export function MarkdownRenderer({ content }: { content: string | null | undefined }) {
    if (!content) return null;

    return (
        <ReactMarkdown
            remarkPlugins={[remarkGfm]}
            components={{
                h1: ({ children }) => <h1 className="text-3xl md:text-4xl font-bold text-white mt-8 mb-4">{children}</h1>,
                h2: ({ children }) => <h2 className="text-2xl md:text-3xl font-bold text-white mt-8 mb-4">{children}</h2>,
                h3: ({ children }) => <h3 className="text-xl md:text-2xl font-semibold text-white mt-6 mb-3">{children}</h3>,
                h4: ({ children }) => <h4 className="text-lg md:text-xl font-semibold text-white mt-6 mb-3">{children}</h4>,
                h5: ({ children }) => <h5 className="text-base md:text-lg font-medium text-white mt-4 mb-2">{children}</h5>,
                h6: ({ children }) => <h6 className="text-sm md:text-base font-medium text-white mt-4 mb-2">{children}</h6>,
                p: ({ children }) => <p className="text-base leading-relaxed md:text-lg mb-4">{children}</p>,
                ul: ({ children }) => <ul className="list-disc pl-6 space-y-2 mb-6">{children}</ul>,
                ol: ({ children }) => <ol className="list-decimal pl-6 space-y-2 mb-6">{children}</ol>,
                li: ({ children }) => <li className="pl-2">{children}</li>,
                blockquote: ({ children }) => (
                    <blockquote className="my-6 border-l-4 border-cyan-500/50 bg-slate-900/50 py-4 pl-6 pr-4 rounded-r-lg italic text-slate-300">
                        {children}
                    </blockquote>
                ),
                code: ({ className, children, ...props }) => {
                    const match = /language-(\w+)/.exec(className || '');
                    const isInline = !match && !String(children).includes('\n');
                    if (isInline) {
                        return <code className="rounded bg-slate-800 px-1.5 py-0.5 text-sm font-mono text-cyan-400" {...props}>{children}</code>
                    }
                    return (
                        <div className="my-6 overflow-x-auto rounded-xl bg-slate-900 border border-slate-800 p-4">
                            <code className={cn("text-sm font-mono text-cyan-300", className)} {...props}>
                                {children}
                            </code>
                        </div>
                    )
                },
                a: ({ href, children }) => (
                    <a
                        href={href}
                        target="_blank"
                        rel="noopener noreferrer"
                        className="text-cyan-400 underline underline-offset-4 decoration-cyan-400/40 hover:decoration-cyan-400 transition-colors"
                    >
                        {children}
                    </a>
                ),
                img: ({ src, alt }) => {
                    if (!src) return null;
                    // Handle Strapi relative URLs
                    const finalSrc = src.startsWith("http") ? src : getStrapiMediaUrl(src);

                    return (
                        <figure className="my-8">
                            <div className="relative aspect-video overflow-hidden rounded-xl border border-slate-800 bg-slate-900">
                                <Image
                                    src={finalSrc || ""}
                                    alt={alt || ""}
                                    fill
                                    className="object-cover"
                                    sizes="(max-width: 768px) 100vw, 800px"
                                />
                            </div>
                            {alt && <figcaption className="mt-3 text-center text-sm text-slate-500">{alt}</figcaption>}
                        </figure>
                    );
                }
            }}
        >
            {content}
        </ReactMarkdown>
    );
}
