"use client";

import { useState, useEffect, useCallback, useMemo } from "react";
import { useTranslations } from "next-intl";
import Link from "next/link";
import { usePathname } from "@/core/lib/i18n/navigation";
import { motion, AnimatePresence, type Variants } from "framer-motion";
import { Menu, X, ChevronRight } from "lucide-react";
import { useLocale } from "next-intl";
import Image from "next/image";
import { MotionButton } from "@/components/ui/button";

export function Navbar() {
  const t = useTranslations("nav");
  const pathname = usePathname();
  const locale = useLocale();
  const [isOpen, setIsOpen] = useState(false);
  const [scrolled, setScrolled] = useState(false);
  const [activeSection, setActiveSection] = useState("hero");

  const navItems = useMemo(
    () => [
      { key: "home", sectionId: "hero" },
      { key: "about", sectionId: "whyus" },
      { key: "services", sectionId: "services" },
      { key: "portfolio", sectionId: "portfolio" },
      { key: "contact", sectionId: "contact" },
    ],
    []
  );

  // Handle scroll effects
  useEffect(() => {
    const handleScroll = () => {
      setScrolled(window.scrollY > 20);

      // Track active section
      const sections = navItems.map((item) => item.sectionId);
      for (const sectionId of [...sections].reverse()) {
        const element = document.getElementById(sectionId);
        if (element) {
          const rect = element.getBoundingClientRect();
          if (rect.top <= 150) {
            setActiveSection(sectionId);
            break;
          }
        }
      }
    };

    window.addEventListener("scroll", handleScroll);
    return () => window.removeEventListener("scroll", handleScroll);
  }, [navItems]);

  // Compute href to a section on the landing page for the current locale
  const getSectionHref = useCallback(
    (sectionId: string) => `/${locale}/#${sectionId}`,
    [locale]
  );

  const isOnHomePage = pathname === "/";

  // Smooth scroll to section when already on the landing page
  const scrollToSection = useCallback(
    (sectionId: string) => {
      const element = document.getElementById(sectionId);
      if (element) {
        const offset = 80;
        const elementPosition = element.getBoundingClientRect().top;
        const offsetPosition = elementPosition + window.pageYOffset - offset;

        window.scrollTo({
          top: offsetPosition,
          behavior: "smooth",
        });
      }
      setIsOpen(false);
    },
    []
  );

  // Handle navigation click:
  // - If we are on the home page, smoothly scroll to the section
  // - If we are on another page, let the browser/Next.js navigate to the landing page section
  const handleNavClick = useCallback(
    (e: React.MouseEvent<HTMLAnchorElement>, sectionId: string) => {
      if (isOnHomePage) {
        e.preventDefault();
        scrollToSection(sectionId);
      } else {
        setIsOpen(false);
        // Allow default navigation to /{locale}/#sectionId
      }
    },
    [isOnHomePage, scrollToSection]
  );

  // Animation variants with proper types
  const navVariants: Variants = {
    hidden: { y: -100, opacity: 0 },
    visible: {
      y: 0,
      opacity: 1,
      transition: {
        type: "spring" as const,
        stiffness: 100,
        damping: 20,
      },
    },
  };

  const mobileMenuVariants: Variants = {
    closed: {
      opacity: 0,
      x: "100%",
      transition: {
        type: "spring" as const,
        stiffness: 400,
        damping: 40,
      },
    },
    open: {
      opacity: 1,
      x: 0,
      transition: {
        type: "spring" as const,
        stiffness: 400,
        damping: 40,
        staggerChildren: 0.1,
        delayChildren: 0.2,
      },
    },
  };

  const mobileItemVariants: Variants = {
    closed: { opacity: 0, x: 50 },
    open: {
      opacity: 1,
      x: 0,
      transition: {
        type: "spring" as const,
        stiffness: 300,
        damping: 24,
      },
    },
  };

  return (
    <>
      <motion.nav
        variants={navVariants}
        initial="hidden"
        animate="visible"
        className={`fixed top-0 z-50 w-full transition-all duration-500 ${scrolled ? "py-2" : "py-4"
          }`}
      >
        {/* Glassmorphism background */}
        <div
          className={`absolute inset-0 transition-all duration-500 ${scrolled
            ? "bg-black/40 backdrop-blur-xl border-b border-white/10 shadow-lg shadow-black/10"
            : "bg-transparent"
            }`}
        />

        {/* Animated gradient border */}
        <motion.div
          className={`absolute bottom-0 left-0 right-0 h-px bg-linear-to-r from-transparent via-cyan-500/50 to-transparent transition-opacity duration-500 ${scrolled ? "opacity-100" : "opacity-0"
            }`}
          animate={{
            backgroundPosition: ["0% 50%", "100% 50%", "0% 50%"],
          }}
          transition={{
            duration: 5,
            repeat: Infinity,
            ease: "linear",
          }}
        />

        <div className="container relative flex items-center justify-between px-4 md:px-6">
          {/* Logo */}
          <motion.div
            whileHover={{ scale: 1.02 }}
            whileTap={{ scale: 0.98 }}
            className="relative z-10"
          >
            <Link href="/" className="flex items-center space-x-2 group">
              <div className="relative">
                {/* Logo glow effect */}
                <motion.div
                  initial={{ opacity: 0, scale: 0.8 }}
                  animate={{
                    opacity: [0.5, 1, 0.5],
                    scale: [0.8, 1.2, 0.8],
                  }}
                  transition={{
                    duration: 3,
                    repeat: Infinity,
                    ease: "easeInOut",
                  }}
                  className="absolute -inset-2 bg-cyan-500/20 rounded-full blur-xl"
                />
                <Image
                  src="/images/logo.png"
                  alt="YSLootah Tech"
                  width={scrolled ? 160 : 180}
                  height={scrolled ? 160 : 180}
                  className="relative transition-all duration-300"
                />
              </div>
            </Link>
          </motion.div>

          {/* Desktop Navigation - centered */}
          <div className="hidden md:flex flex-1 items-center justify-center gap-1 relative z-10">
            {/* Glassmorphism pill container */}
            <div className="flex items-center gap-1 px-2 py-2 rounded-full bg-white/5 backdrop-blur-md border border-white/10">
              {navItems.map((item) => (
                <motion.a
                  key={item.key}
                  href={getSectionHref(item.sectionId)}
                  onClick={(e) => handleNavClick(e, item.sectionId)}
                  initial={{ scale: 1 }}
                  whileHover={{ scale: 1.05 }}
                  transition={{ type: "spring", stiffness: 400, damping: 10 }}
                  className={`relative px-4 py-2 text-sm font-medium rounded-full transition-all duration-300 ${activeSection === item.sectionId
                    ? "text-white"
                    : "text-white/70 hover:text-white"
                    }`}
                >
                  {/* Active indicator */}
                  {activeSection === item.sectionId && (
                    <motion.div
                      layoutId="activeSection"
                      className="absolute inset-0 rounded-full bg-linear-to-r from-cyan-500/30 to-blue-500/30 border border-cyan-500/30"
                      transition={{
                        type: "spring",
                        stiffness: 380,
                        damping: 30,
                      }}
                    />
                  )}

                  {/* Hover glow */}
                  <motion.div
                    className="absolute inset-0 rounded-full bg-white/5 opacity-0"
                    whileHover={{ opacity: 1 }}
                    transition={{ duration: 0.2 }}
                  />

                  <span className="relative z-10">{t(item.key)}</span>
                </motion.a>
              ))}
            </div>
          </div>

          {/* Desktop CTA Button */}
          <div className="hidden md:block ml-4 relative z-10">
            <MotionButton
              variant="gradient"
              size="default"
              iconRight={<ChevronRight className="w-4 h-4" />}
              onClick={(e) => {
                if (isOnHomePage) {
                  e.preventDefault();
                  scrollToSection("contact");
                } else {
                  window.location.assign(getSectionHref("contact"));
                }
              }}
            >
              Get Started
            </MotionButton>
          </div>

          {/* Mobile Menu Button */}
          <motion.button
            whileHover={{ scale: 1.1 }}
            whileTap={{ scale: 0.9 }}
            onClick={() => setIsOpen(!isOpen)}
            className="md:hidden relative z-10 p-2 rounded-full bg-white/10 backdrop-blur-md border border-white/10"
          >
            <AnimatePresence mode="wait">
              {isOpen ? (
                <motion.div
                  key="close"
                  initial={{ rotate: -90, opacity: 0 }}
                  animate={{ rotate: 0, opacity: 1 }}
                  exit={{ rotate: 90, opacity: 0 }}
                  transition={{ duration: 0.2 }}
                >
                  <X className="h-5 w-5 text-white" />
                </motion.div>
              ) : (
                <motion.div
                  key="menu"
                  initial={{ rotate: 90, opacity: 0 }}
                  animate={{ rotate: 0, opacity: 1 }}
                  exit={{ rotate: -90, opacity: 0 }}
                  transition={{ duration: 0.2 }}
                >
                  <Menu className="h-5 w-5 text-white" />
                </motion.div>
              )}
            </AnimatePresence>
          </motion.button>
        </div>
      </motion.nav>

      {/* Mobile Menu Overlay */}
      <AnimatePresence>
        {isOpen && (
          <>
            {/* Backdrop */}
            <motion.div
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
              exit={{ opacity: 0 }}
              onClick={() => setIsOpen(false)}
              className="fixed inset-0 z-40 bg-black/60 backdrop-blur-sm md:hidden"
            />

            {/* Mobile Menu Panel */}
            <motion.div
              variants={mobileMenuVariants}
              initial="closed"
              animate="open"
              exit="closed"
              className="fixed top-0 right-0 bottom-0 z-50 w-[85%] max-w-100 md:hidden"
            >
              {/* Glassmorphism panel */}
              <div className="h-full bg-linear-to-b from-gray-900/95 to-black/95 backdrop-blur-xl border-l border-white/10">
                {/* Decorative elements */}
                <div className="absolute top-0 right-0 w-64 h-64 bg-cyan-500/10 rounded-full blur-3xl" />
                <div className="absolute bottom-0 left-0 w-48 h-48 bg-blue-500/10 rounded-full blur-3xl" />

                {/* Close button */}
                <div className="flex justify-end p-4">
                  <motion.button
                    whileHover={{ scale: 1.1, rotate: 90 }}
                    whileTap={{ scale: 0.9 }}
                    onClick={() => setIsOpen(false)}
                    className="p-2 rounded-full bg-white/10 border border-white/10"
                  >
                    <X className="h-5 w-5 text-white" />
                  </motion.button>
                </div>

                {/* Menu Items */}
                <div className="relative flex flex-col px-6 py-4 space-y-2">
                  {navItems.map((item, index) => (
                    <motion.a
                      key={item.key}
                      variants={mobileItemVariants}
                      href={getSectionHref(item.sectionId)}
                      onClick={(e) => handleNavClick(e, item.sectionId)}
                      className={`group relative flex items-center gap-4 px-4 py-4 rounded-2xl transition-all duration-300 ${activeSection === item.sectionId
                        ? "bg-linear-to-r from-cyan-500/20 to-blue-500/20 border border-cyan-500/30"
                        : "hover:bg-white/5"
                        }`}
                    >
                      {/* Index number */}
                      <span className="text-xs font-mono text-white/30">
                        0{index + 1}
                      </span>

                      {/* Link text */}
                      <span
                        className={`text-lg font-medium transition-colors ${activeSection === item.sectionId
                          ? "text-white"
                          : "text-white/70 group-hover:text-white"
                          }`}
                      >
                        {t(item.key)}
                      </span>

                      {/* Arrow icon */}
                      <ChevronRight
                        className={`ml-auto w-5 h-5 transition-all duration-300 ${activeSection === item.sectionId
                          ? "text-cyan-400 translate-x-0 opacity-100"
                          : "text-white/30 -translate-x-2 opacity-0 group-hover:translate-x-0 group-hover:opacity-100"
                          }`}
                      />
                    </motion.a>
                  ))}

                  {/* CTA Button */}
                  <motion.div variants={mobileItemVariants} className="pt-4">
                    <MotionButton
                      variant="gradient"
                      size="xl"
                      className="w-full rounded-2xl"
                      iconRight={<ChevronRight className="w-5 h-5" />}
                      onClick={(e) => {
                        if (isOnHomePage) {
                          e.preventDefault();
                          scrollToSection("contact");
                        } else {
                          window.location.assign(getSectionHref("contact"));
                        }
                      }}
                    >
                      Get Started
                    </MotionButton>
                  </motion.div>
                </div>
              </div>
            </motion.div>
          </>
        )}
      </AnimatePresence>
    </>
  );
}
