"use client";

import * as React from "react";
import { Slot } from "@radix-ui/react-slot";
import { cva, type VariantProps } from "class-variance-authority";
import { motion, type HTMLMotionProps } from "motion/react";

import { cn } from "@/core/lib/utils";

const buttonVariants = cva(
  "group relative inline-flex items-center cursor-pointer justify-center gap-2 whitespace-nowrap text-sm font-medium transition-all duration-300 focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-cyan-500/50 focus-visible:ring-offset-2 focus-visible:ring-offset-slate-950 disabled:pointer-events-none disabled:opacity-50 [&_svg]:pointer-events-none [&_svg]:shrink-0 overflow-hidden",
  {
    variants: {
      variant: {
        // Primary gradient button - cyan to blue with shine effect
        gradient:
          "rounded-full bg-linear-to-r from-cyan-500 to-blue-600 text-white font-semibold shadow-lg shadow-cyan-500/25 hover:shadow-cyan-500/40 hover:shadow-xl",
        // Gradient outline - transparent bg with gradient border glow
        "gradient-outline":
          "rounded-full border border-cyan-500/50 bg-transparent text-white font-semibold hover:border-cyan-400 hover:bg-cyan-500/10",
        // Solid outline
        outline:
          "rounded-full border border-slate-600 bg-transparent text-white font-medium hover:border-slate-500 hover:bg-white/5",
        // Ghost - minimal styling
        ghost:
          "rounded-lg text-slate-300 hover:text-white hover:bg-white/10",
        // Secondary - slate background
        secondary:
          "rounded-full bg-slate-800 text-white font-medium hover:bg-slate-700",
        // White solid - for dark backgrounds
        solid:
          "rounded-xl bg-white text-slate-900 font-semibold hover:bg-blue-500 hover:text-white",
        // Link style
        link: "text-cyan-400 underline-offset-4 hover:underline hover:text-cyan-300",
      },
      size: {
        sm: "h-9 px-4 text-sm [&_svg]:size-4",
        default: "h-11 px-6 text-sm [&_svg]:size-4",
        lg: "h-12 px-8 text-base [&_svg]:size-5",
        xl: "h-14 px-10 text-lg [&_svg]:size-5",
        "2xl": "h-16 px-12 text-lg [&_svg]:size-6",
        icon: "h-10 w-10 rounded-full",
        "icon-sm": "h-8 w-8 rounded-full",
        "icon-lg": "h-12 w-12 rounded-full",
      },
    },
    defaultVariants: {
      variant: "gradient",
      size: "default",
    },
  }
);

// Shine effect component for gradient buttons
function ShineEffect({ className }: { className?: string }) {
  return (
    <motion.span
      className={cn(
        "absolute inset-0 bg-linear-to-r from-transparent via-white/25 to-transparent -skew-x-12 pointer-events-none",
        className
      )}
      initial={{ x: "-200%" }}
      whileHover={{ x: "200%" }}
      transition={{
        duration: 0.6,
        ease: "easeInOut",
      }}
    />
  );
}

// Glow effect for gradient buttons
function GlowEffect() {
  return (
    <span className="absolute inset-0 rounded-[inherit] opacity-0 group-hover:opacity-100 transition-opacity duration-300 bg-linear-to-r from-cyan-500/20 to-blue-500/20 blur-xl -z-10" />
  );
}

// Border glow for outline variants
function BorderGlow() {
  return (
    <span className="absolute inset-0 rounded-[inherit] opacity-0 group-hover:opacity-100 transition-opacity duration-300">
      <span className="absolute -inset-px rounded-[inherit] bg-linear-to-r from-cyan-500/50 via-blue-500/50 to-cyan-500/50 blur-sm" />
    </span>
  );
}

export interface ButtonProps
  extends React.ButtonHTMLAttributes<HTMLButtonElement>,
  VariantProps<typeof buttonVariants> {
  asChild?: boolean;
  /** Enable shine animation on hover (auto-enabled for gradient variant) */
  shine?: boolean;
  /** Enable glow effect on hover (auto-enabled for gradient variant) */
  glow?: boolean;
  /** Icon to show at the start */
  iconLeft?: React.ReactNode;
  /** Icon to show at the end (animates on hover) */
  iconRight?: React.ReactNode;
  /** Enable arrow animation for iconRight on hover */
  animateIcon?: boolean;
}

const Button = React.forwardRef<HTMLButtonElement, ButtonProps>(
  (
    {
      className,
      variant,
      size,
      asChild = false,
      shine,
      glow,
      iconLeft,
      iconRight,
      animateIcon = true,
      children,
      ...props
    },
    ref
  ) => {
    // Auto-enable shine and glow for gradient variant
    const showShine = shine ?? variant === "gradient";
    const showGlow = glow ?? variant === "gradient";
    const showBorderGlow = variant === "gradient-outline";

    if (asChild) {
      return (
        <Slot
          className={cn(buttonVariants({ variant, size, className }))}
          ref={ref}
          {...props}
        >
          {children}
        </Slot>
      );
    }

    return (
      <button
        className={cn(buttonVariants({ variant, size, className }))}
        ref={ref}
        {...props}
      >
        {/* Effects */}
        {showGlow && <GlowEffect />}
        {showBorderGlow && <BorderGlow />}

        {/* Content wrapper */}
        <span className="relative z-10 flex items-center gap-2">
          {iconLeft && (
            <span className="shrink-0">{iconLeft}</span>
          )}
          {children}
          {iconRight && (
            <span
              className={cn(
                "shrink-0 transition-transform duration-300",
                animateIcon && "group-hover:translate-x-1"
              )}
            >
              {iconRight}
            </span>
          )}
        </span>

        {/* Shine effect - rendered last to be on top */}
        {showShine && <ShineEffect />}
      </button>
    );
  }
);
Button.displayName = "Button";

// Motion-enhanced button for advanced animations
export interface MotionButtonProps
  extends Omit<HTMLMotionProps<"button">, "ref">,
  VariantProps<typeof buttonVariants> {
  /** Enable shine animation on hover */
  shine?: boolean;
  /** Enable glow effect on hover */
  glow?: boolean;
  /** Icon to show at the start */
  iconLeft?: React.ReactNode;
  /** Icon to show at the end */
  iconRight?: React.ReactNode;
  /** Enable arrow animation for iconRight on hover */
  animateIcon?: boolean;
}

const MotionButton = React.forwardRef<HTMLButtonElement, MotionButtonProps>(
  (
    {
      className,
      variant,
      size,
      shine,
      glow,
      iconLeft,
      iconRight,
      animateIcon = true,
      children,
      whileHover = { scale: 1.02 },
      whileTap = { scale: 0.98 },
      ...props
    },
    ref
  ) => {
    const showShine = shine ?? variant === "gradient";
    const showGlow = glow ?? variant === "gradient";
    const showBorderGlow = variant === "gradient-outline";

    return (
      <motion.button
        className={cn(buttonVariants({ variant, size, className }))}
        ref={ref}
        whileHover={whileHover}
        whileTap={whileTap}
        {...props}
      >
        {/* Effects */}
        {showGlow && <GlowEffect />}
        {showBorderGlow && <BorderGlow />}

        {/* Content wrapper */}
        <span className="relative z-10 flex items-center gap-2">
          {iconLeft && (
            <span className="shrink-0">{iconLeft}</span>
          )}
          {children as React.ReactNode}
          {iconRight && (
            <span
              className={cn(
                "shrink-0 transition-transform duration-300",
                animateIcon && "group-hover:translate-x-1"
              )}
            >
              {iconRight}
            </span>
          )}
        </span>

        {/* Shine effect */}
        {showShine && <ShineEffect />}
      </motion.button>
    );
  }
);
MotionButton.displayName = "MotionButton";

export { Button, MotionButton, buttonVariants };
