"use client";

import { useMotionValue, motion, useMotionTemplate } from "motion/react";
import React, { useCallback, type HTMLAttributes } from "react";
import { cn } from "@/core/lib/utils";

export interface CardSpotlightProps extends HTMLAttributes<HTMLDivElement> {
  // Use `any` here to avoid ReactNode type incompatibilities across React versions.
  // Consumers should still pass valid React children.
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  children?: any;
  radius?: number;
  color?: string;
  className?: string;
  /** Whether to show the spotlight gradient on hover */
  showGradient?: boolean;
}

export function CardSpotlight({
  children,
  radius = 350,
  color = "#262626",
  className,
  showGradient = true,
  ...props
}: CardSpotlightProps) {
  const mouseX = useMotionValue(0);
  const mouseY = useMotionValue(0);

  const handleMouseMove = useCallback(
    (event: React.MouseEvent<HTMLDivElement>) => {
      const { currentTarget, clientX, clientY } = event;
      const { left, top } = currentTarget.getBoundingClientRect();
      mouseX.set(clientX - left);
      mouseY.set(clientY - top);
    },
    [mouseX, mouseY]
  );

  return (
    <div
      className={cn(
        "group/spotlight relative rounded-xl overflow-hidden",
        className
      )}
      onMouseMove={handleMouseMove}
      {...props}
    >
      {showGradient && (
        <motion.div
          className="pointer-events-none absolute -inset-px z-0 rounded-[inherit] opacity-0 transition-opacity duration-300 group-hover/spotlight:opacity-100"
          style={{
            background: useMotionTemplate`
              radial-gradient(
                ${radius}px circle at ${mouseX}px ${mouseY}px,
                ${color},
                transparent 80%
              )
            `,
          }}
        />
      )}
      {/* Inner glow border effect */}
      <motion.div
        className="pointer-events-none absolute inset-0 z-0 rounded-[inherit] opacity-0 transition-opacity duration-300 group-hover/spotlight:opacity-100"
        style={{
          background: useMotionTemplate`
            radial-gradient(
              ${radius * 0.6}px circle at ${mouseX}px ${mouseY}px,
              rgba(59, 130, 246, 0.15),
              transparent 60%
            )
          `,
        }}
      />
      {children}
    </div>
  );
}

/**
 * A variant that wraps `motion.div` for cards that need framer-motion animations.
 *
 * We base the props on `motion.div` itself to avoid direct dependency on `MotionProps`,
 * which can introduce `children` typing conflicts across React versions.
 */
type BaseMotionDivProps = React.ComponentProps<typeof motion.div>;

export interface MotionCardSpotlightProps extends Omit<BaseMotionDivProps, "children"> {
  // Use `any` here to avoid ReactNode type incompatibilities across React versions.
  // Consumers should still pass valid React children.
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  children?: any;
  radius?: number;
  color?: string;
  showGradient?: boolean;
}

export const MotionCardSpotlight = React.forwardRef<HTMLDivElement, MotionCardSpotlightProps>(
  ({ children, radius = 350, color = "#262626", showGradient = true, className, ...props }, ref) => {
    const mouseX = useMotionValue(0);
    const mouseY = useMotionValue(0);

    const handleMouseMove = useCallback(
      (event: React.MouseEvent<HTMLDivElement>) => {
        const { currentTarget, clientX, clientY } = event;
        const { left, top } = currentTarget.getBoundingClientRect();
        mouseX.set(clientX - left);
        mouseY.set(clientY - top);
      },
      [mouseX, mouseY]
    );

    return (
      <motion.div
        ref={ref}
        className={cn(
          "group/spotlight relative rounded-xl overflow-hidden",
          className
        )}
        onMouseMove={handleMouseMove}
        {...props}
      >
        {showGradient && (
          <motion.div
            className="pointer-events-none absolute -inset-px z-0 rounded-[inherit] opacity-0 transition-opacity duration-300 group-hover/spotlight:opacity-100"
            style={{
              background: useMotionTemplate`
                radial-gradient(
                  ${radius}px circle at ${mouseX}px ${mouseY}px,
                  ${color},
                  transparent 80%
                )
              `,
            }}
          />
        )}
        {/* Inner glow border effect */}
        <motion.div
          className="pointer-events-none absolute inset-0 z-0 rounded-[inherit] opacity-0 transition-opacity duration-300 group-hover/spotlight:opacity-100"
          style={{
            background: useMotionTemplate`
              radial-gradient(
                ${radius * 0.6}px circle at ${mouseX}px ${mouseY}px,
                rgba(59, 130, 246, 0.15),
                transparent 60%
              )
            `,
          }}
        />
        {children}
      </motion.div>
    );
  }
);

MotionCardSpotlight.displayName = "MotionCardSpotlight";
