"use client";

import { useEffect, useState, useCallback, useRef } from "react";
import { motion, useMotionValue, useSpring, AnimatePresence, MotionValue } from "framer-motion";

// Particle Trail Component - hooks are called at top level
function ParticleTrail({ 
  cursorX, 
  cursorY, 
  isVisible 
}: { 
  cursorX: MotionValue<number>; 
  cursorY: MotionValue<number>; 
  isVisible: boolean;
}) {
  // Create springs for each particle at component level
  const p1x = useSpring(cursorX, { damping: 40, stiffness: 150, mass: 0.5 });
  const p1y = useSpring(cursorY, { damping: 40, stiffness: 150, mass: 0.5 });
  const p2x = useSpring(cursorX, { damping: 45, stiffness: 130, mass: 0.6 });
  const p2y = useSpring(cursorY, { damping: 45, stiffness: 130, mass: 0.6 });
  const p3x = useSpring(cursorX, { damping: 50, stiffness: 110, mass: 0.7 });
  const p3y = useSpring(cursorY, { damping: 50, stiffness: 110, mass: 0.7 });
  const p4x = useSpring(cursorX, { damping: 55, stiffness: 90, mass: 0.8 });
  const p4y = useSpring(cursorY, { damping: 55, stiffness: 90, mass: 0.8 });
  const p5x = useSpring(cursorX, { damping: 60, stiffness: 70, mass: 0.9 });
  const p5y = useSpring(cursorY, { damping: 60, stiffness: 70, mass: 0.9 });

  const particles = [
    { x: p1x, y: p1y, scale: 1, opacity: 0.5 },
    { x: p2x, y: p2y, scale: 0.85, opacity: 0.4 },
    { x: p3x, y: p3y, scale: 0.7, opacity: 0.3 },
    { x: p4x, y: p4y, scale: 0.55, opacity: 0.2 },
    { x: p5x, y: p5y, scale: 0.4, opacity: 0.1 },
  ];

  return (
    <div className="fixed inset-0 pointer-events-none z-[9996]">
      {particles.map((particle, i) => (
        <motion.div
          key={i}
          className="absolute w-1.5 h-1.5 rounded-full bg-gradient-to-r from-sky-400 to-blue-500"
          style={{
            x: particle.x,
            y: particle.y,
            translateX: "-50%",
            translateY: "-50%",
            scale: particle.scale,
            opacity: isVisible ? particle.opacity : 0,
          }}
        />
      ))}
    </div>
  );
}

interface CursorState {
  isHovering: boolean;
  isClicking: boolean;
  cursorType: "default" | "link" | "button" | "text";
  cursorText: string;
}

export function CustomCursor() {
  const [isVisible, setIsVisible] = useState(false);
  const [isMobile, setIsMobile] = useState(true);
  const [cursorState, setCursorState] = useState<CursorState>({
    isHovering: false,
    isClicking: false,
    cursorType: "default",
    cursorText: "",
  });

  // Mouse position with motion values for smooth tracking
  const cursorX = useMotionValue(-100);
  const cursorY = useMotionValue(-100);
  
  // Spring configurations for different elements
  const springConfig = { damping: 25, stiffness: 400, mass: 0.5 };
  const trailSpringConfig = { damping: 30, stiffness: 200, mass: 0.8 };
  const glowSpringConfig = { damping: 35, stiffness: 150, mass: 1 };

  // Smooth cursor positions with springs
  const smoothX = useSpring(cursorX, springConfig);
  const smoothY = useSpring(cursorY, springConfig);
  
  // Trail follows with more delay
  const trailX = useSpring(cursorX, trailSpringConfig);
  const trailY = useSpring(cursorY, trailSpringConfig);
  
  // Glow follows with even more delay
  const glowX = useSpring(cursorX, glowSpringConfig);
  const glowY = useSpring(cursorY, glowSpringConfig);

  // Track cursor trail positions for particle effect
  const trailPositions = useRef<{ x: number; y: number; id: number }[]>([]);
  const trailIdCounter = useRef(0);

  const handleMouseMove = useCallback((e: MouseEvent) => {
    cursorX.set(e.clientX);
    cursorY.set(e.clientY);
    setIsVisible(true);

    // Add to trail
    trailIdCounter.current += 1;
    trailPositions.current.push({
      x: e.clientX,
      y: e.clientY,
      id: trailIdCounter.current,
    });
    
    // Keep only last 8 positions
    if (trailPositions.current.length > 8) {
      trailPositions.current.shift();
    }
  }, [cursorX, cursorY]);

  const handleMouseDown = useCallback(() => {
    setCursorState((prev) => ({ ...prev, isClicking: true }));
  }, []);

  const handleMouseUp = useCallback(() => {
    setCursorState((prev) => ({ ...prev, isClicking: false }));
  }, []);

  const handleMouseEnter = useCallback(() => {
    setIsVisible(true);
  }, []);

  const handleMouseLeave = useCallback(() => {
    setIsVisible(false);
  }, []);

  // Check for interactive elements
  const handleElementHover = useCallback((e: MouseEvent) => {
    const target = e.target as HTMLElement;
    
    // Check if hovering over interactive elements
    const isLink = target.closest("a") !== null;
    const isButton = target.closest("button") !== null || target.closest('[role="button"]') !== null;
    const isInput = target.tagName === "INPUT" || target.tagName === "TEXTAREA";
    
    // Get custom cursor text if any
    const cursorText = target.closest("[data-cursor-text]")?.getAttribute("data-cursor-text") || "";
    
    if (isLink) {
      setCursorState((prev) => ({
        ...prev,
        isHovering: true,
        cursorType: "link",
        cursorText,
      }));
    } else if (isButton) {
      setCursorState((prev) => ({
        ...prev,
        isHovering: true,
        cursorType: "button",
        cursorText,
      }));
    } else if (isInput) {
      setCursorState((prev) => ({
        ...prev,
        isHovering: true,
        cursorType: "text",
        cursorText: "",
      }));
    } else {
      setCursorState((prev) => ({
        ...prev,
        isHovering: false,
        cursorType: "default",
        cursorText: "",
      }));
    }
  }, []);

  useEffect(() => {
    // Check if mobile/touch device
    const checkMobile = () => {
      const isTouchDevice = "ontouchstart" in window || navigator.maxTouchPoints > 0;
      const isSmallScreen = window.innerWidth < 768;
      setIsMobile(isTouchDevice || isSmallScreen);
    };

    checkMobile();
    window.addEventListener("resize", checkMobile);

    if (!isMobile) {
      // Hide default cursor
      document.body.style.cursor = "none";
      
      // Add event listeners
      window.addEventListener("mousemove", handleMouseMove);
      window.addEventListener("mousemove", handleElementHover);
      window.addEventListener("mousedown", handleMouseDown);
      window.addEventListener("mouseup", handleMouseUp);
      document.addEventListener("mouseenter", handleMouseEnter);
      document.addEventListener("mouseleave", handleMouseLeave);

      // Add cursor:none to all interactive elements
      const style = document.createElement("style");
      style.id = "custom-cursor-styles";
      style.textContent = `
        * { cursor: none !important; }
        a, button, [role="button"], input, textarea, select { cursor: none !important; }
      `;
      document.head.appendChild(style);
    }

    return () => {
      document.body.style.cursor = "";
      window.removeEventListener("mousemove", handleMouseMove);
      window.removeEventListener("mousemove", handleElementHover);
      window.removeEventListener("mousedown", handleMouseDown);
      window.removeEventListener("mouseup", handleMouseUp);
      document.removeEventListener("mouseenter", handleMouseEnter);
      document.removeEventListener("mouseleave", handleMouseLeave);
      window.removeEventListener("resize", checkMobile);
      
      const style = document.getElementById("custom-cursor-styles");
      if (style) style.remove();
    };
  }, [isMobile, handleMouseMove, handleElementHover, handleMouseDown, handleMouseUp, handleMouseEnter, handleMouseLeave]);

  // Don't render on mobile
  if (isMobile) return null;

  const { isHovering, isClicking, cursorType, cursorText } = cursorState;

  return (
    <>
      {/* Main glow effect - largest, most delayed */}
      <motion.div
        className="fixed top-0 left-0 pointer-events-none z-[9997] mix-blend-screen"
        style={{
          x: glowX,
          y: glowY,
          translateX: "-50%",
          translateY: "-50%",
        }}
      >
        <motion.div
          className="rounded-full"
          animate={{
            width: isHovering ? 120 : 80,
            height: isHovering ? 120 : 80,
            opacity: isVisible ? (isHovering ? 0.15 : 0.1) : 0,
          }}
          transition={{ duration: 0.4, ease: "easeOut" }}
          style={{
            background: `radial-gradient(circle, rgba(91, 155, 213, 0.4) 0%, rgba(56, 189, 248, 0.2) 40%, transparent 70%)`,
            filter: "blur(20px)",
          }}
        />
      </motion.div>

      {/* Outer ring - follows with delay */}
      <motion.div
        className="fixed top-0 left-0 pointer-events-none z-[9998]"
        style={{
          x: trailX,
          y: trailY,
          translateX: "-50%",
          translateY: "-50%",
        }}
      >
        <motion.div
          className="rounded-full border backdrop-blur-[2px]"
          animate={{
            width: isHovering ? (cursorType === "button" ? 64 : 56) : 40,
            height: isHovering ? (cursorType === "button" ? 64 : 56) : 40,
            borderColor: isHovering 
              ? "rgba(56, 189, 248, 0.6)" 
              : "rgba(91, 155, 213, 0.4)",
            backgroundColor: isHovering 
              ? "rgba(56, 189, 248, 0.08)" 
              : "rgba(91, 155, 213, 0.04)",
            scale: isClicking ? 0.85 : 1,
            opacity: isVisible ? 1 : 0,
          }}
          transition={{
            duration: 0.3,
            ease: [0.23, 1, 0.32, 1],
          }}
          style={{
            boxShadow: isHovering 
              ? "0 0 20px rgba(56, 189, 248, 0.3), inset 0 0 20px rgba(56, 189, 248, 0.1)"
              : "0 0 10px rgba(91, 155, 213, 0.2)",
          }}
        />
      </motion.div>

      {/* Inner dot - direct follow */}
      <motion.div
        className="fixed top-0 left-0 pointer-events-none z-[9999]"
        style={{
          x: smoothX,
          y: smoothY,
          translateX: "-50%",
          translateY: "-50%",
        }}
      >
        <motion.div
          className="rounded-full"
          animate={{
            width: isHovering 
              ? (cursorType === "text" ? 2 : cursorType === "button" ? 8 : 6) 
              : 8,
            height: isHovering 
              ? (cursorType === "text" ? 24 : cursorType === "button" ? 8 : 6) 
              : 8,
            backgroundColor: isHovering 
              ? cursorType === "text" 
                ? "rgba(255, 255, 255, 0.8)"
                : "rgba(56, 189, 248, 1)" 
              : "rgba(91, 155, 213, 1)",
            scale: isClicking ? 0.5 : 1,
            opacity: isVisible ? 1 : 0,
            borderRadius: cursorType === "text" ? "1px" : "50%",
          }}
          transition={{
            duration: 0.15,
            ease: "easeOut",
          }}
          style={{
            boxShadow: "0 0 12px rgba(91, 155, 213, 0.8), 0 0 24px rgba(56, 189, 248, 0.4)",
          }}
        />
      </motion.div>

      {/* Cursor text label */}
      <AnimatePresence>
        {cursorText && isHovering && (
          <motion.div
            className="fixed top-0 left-0 pointer-events-none z-[10000]"
            style={{
              x: smoothX,
              y: smoothY,
            }}
            initial={{ opacity: 0, scale: 0.8, y: 20 }}
            animate={{ opacity: 1, scale: 1, y: 28 }}
            exit={{ opacity: 0, scale: 0.8, y: 20 }}
            transition={{ duration: 0.2, ease: "easeOut" }}
          >
            <span 
              className="px-3 py-1.5 text-xs font-medium text-white/90 bg-slate-900/80 backdrop-blur-md rounded-full border border-white/10 whitespace-nowrap"
              style={{
                transform: "translateX(-50%)",
                boxShadow: "0 4px 20px rgba(0, 0, 0, 0.3)",
              }}
            >
              {cursorText}
            </span>
          </motion.div>
        )}
      </AnimatePresence>

      {/* Particle trail effect */}
      <ParticleTrail cursorX={cursorX} cursorY={cursorY} isVisible={isVisible} />

      {/* Magnetic interaction indicator */}
      <AnimatePresence>
        {isHovering && cursorType === "button" && (
          <motion.div
            className="fixed top-0 left-0 pointer-events-none z-[9995]"
            style={{
              x: trailX,
              y: trailY,
              translateX: "-50%",
              translateY: "-50%",
            }}
            initial={{ scale: 0, opacity: 0 }}
            animate={{ scale: 1, opacity: 1 }}
            exit={{ scale: 0, opacity: 0 }}
            transition={{ duration: 0.3 }}
          >
            <motion.div
              className="w-20 h-20 rounded-full border border-sky-400/20"
              animate={{
                scale: [1, 1.2, 1],
                opacity: [0.3, 0.1, 0.3],
              }}
              transition={{
                duration: 1.5,
                repeat: Infinity,
                ease: "easeInOut",
              }}
            />
          </motion.div>
        )}
      </AnimatePresence>
    </>
  );
}
