"use client";
import { useCallback, useEffect, useState } from "react";
import { AnimatePresence, motion } from "framer-motion";
import { cn } from "@/core/lib/utils";

export const FlipWords = ({
  words,
  duration = 3000,
  className,
}: {
  words: string[];
  duration?: number;
  className?: string;
}) => {
  const [currentIndex, setCurrentIndex] = useState(0);
  const [isAnimating, setIsAnimating] = useState(false);

  const startAnimation = useCallback(() => {
    setCurrentIndex((prevIndex) => (prevIndex + 1) % words.length);
    setIsAnimating(true);
  }, [words.length]);

  useEffect(() => {
    if (!isAnimating) {
      const timer = setTimeout(() => {
        startAnimation();
      }, duration);
      return () => clearTimeout(timer);
    }
  }, [isAnimating, duration, startAnimation]);

  return (
    <AnimatePresence
      mode="wait"
      onExitComplete={() => {
        setIsAnimating(false);
      }}
    >
      <motion.div
        key={words[currentIndex]}
        initial={{
          opacity: 0,
          y: 10,
          filter: "blur(8px)",
        }}
        animate={{
          opacity: 1,
          y: 0,
          filter: "blur(0px)",
        }}
        exit={{
          opacity: 0,
          y: -40,
          filter: "blur(8px)",
          position: "absolute",
        }}
        transition={{
          type: "spring",
          stiffness: 100,
          damping: 15,
        }}
        className={cn(
          "inline-block relative text-left",
          className
        )}
      >
        {words[currentIndex].split(" ").map((word, wordIndex) => (
          <motion.span
            key={word + wordIndex}
            initial={{ opacity: 0, y: 10, filter: "blur(8px)" }}
            animate={{ opacity: 1, y: 0, filter: "blur(0px)" }}
            transition={{
              delay: wordIndex * 0.1,
              duration: 0.3,
            }}
            className="inline-block whitespace-nowrap"
          >
            {word.split("").map((letter, letterIndex) => (
              <motion.span
                key={word + letterIndex}
                initial={{ opacity: 0, y: 10, filter: "blur(8px)" }}
                animate={{ opacity: 1, y: 0, filter: "blur(0px)" }}
                transition={{
                  delay: wordIndex * 0.1 + letterIndex * 0.03,
                  duration: 0.2,
                }}
                className="inline-block"
              >
                {letter}
              </motion.span>
            ))}
            <span className="inline-block">&nbsp;</span>
          </motion.span>
        ))}
      </motion.div>
    </AnimatePresence>
  );
};

// Text flip component for the complete phrase animation
export const TextFlip = ({
  phrases,
  duration = 4000,
  className,
  highlightClassName,
}: {
  phrases: { prefix: string; highlight: string }[];
  duration?: number;
  className?: string;
  highlightClassName?: string;
}) => {
  const [currentIndex, setCurrentIndex] = useState(0);
  const [isAnimating, setIsAnimating] = useState(false);

  const startAnimation = useCallback(() => {
    setCurrentIndex((prevIndex) => (prevIndex + 1) % phrases.length);
    setIsAnimating(true);
  }, [phrases.length]);

  useEffect(() => {
    if (!isAnimating) {
      const timer = setTimeout(() => {
        startAnimation();
      }, duration);
      return () => clearTimeout(timer);
    }
  }, [isAnimating, duration, startAnimation]);

  const currentPhrase = phrases[currentIndex];

  return (
    <div className={cn("relative", className)}>
      <AnimatePresence
        mode="wait"
        onExitComplete={() => {
          setIsAnimating(false);
        }}
      >
        <motion.h2
          key={currentIndex}
          className="text-5xl md:text-7xl lg:text-8xl font-bold text-white"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          exit={{ opacity: 0 }}
          transition={{ duration: 0.3 }}
        >
          <motion.span
            initial={{ opacity: 0, x: -20, filter: "blur(10px)" }}
            animate={{ opacity: 1, x: 0, filter: "blur(0px)" }}
            exit={{ opacity: 0, x: 20, filter: "blur(10px)" }}
            transition={{
              type: "spring",
              stiffness: 100,
              damping: 15,
            }}
            className="inline-block"
          >
            {currentPhrase.prefix.split("").map((char, i) => (
              <motion.span
                key={i}
                initial={{ opacity: 0, y: 20, rotateX: 90 }}
                animate={{ opacity: 1, y: 0, rotateX: 0 }}
                exit={{ opacity: 0, y: -20, rotateX: -90 }}
                transition={{
                  delay: i * 0.03,
                  duration: 0.4,
                  type: "spring",
                  stiffness: 150,
                  damping: 20,
                }}
                className="inline-block"
                style={{ transformStyle: "preserve-3d" }}
              >
                {char === " " ? "\u00A0" : char}
              </motion.span>
            ))}
          </motion.span>
          <motion.span
            initial={{ opacity: 0, y: 40, scale: 0.9, filter: "blur(12px)" }}
            animate={{ opacity: 1, y: 0, scale: 1, filter: "blur(0px)" }}
            exit={{ opacity: 0, y: -40, scale: 0.9, filter: "blur(12px)" }}
            transition={{
              delay: 0.3,
              duration: 0.6,
              type: "spring",
              stiffness: 100,
              damping: 15,
            }}
            className={cn(
              "text-transparent bg-clip-text bg-linear-to-r from-secondary via-secondary/80 to-secondary ms-4 inline-block",
              highlightClassName
            )}
            style={{ transformStyle: "preserve-3d" }}
          >
            {currentPhrase.highlight}
          </motion.span>
        </motion.h2>
      </AnimatePresence>
    </div>
  );
};
