"use client";

import { motion, useScroll, useSpring, useTransform } from "framer-motion";

interface ScrollRotateProps {
  children: React.ReactNode;
  /**
   * How many degrees the element rotates per pixel of scroll.
   * Higher = faster rotation. Default: 0.1
   */
  speed?: number;
  /**
   * Spring stiffness — controls how snappy the rotation feels.
   * Default: 80
   */
  stiffness?: number;
  /**
   * Spring damping — controls how quickly the rotation settles when
   * scrolling stops. Lower = more bounce. Default: 20
   */
  damping?: number;
  className?: string;
}

/**
 * Wraps children in a container that rotates while the user scrolls.
 *
 * The rotation angle is proportional to the page scroll offset, so:
 * - Scrolling down → element rotates clockwise
 * - Scrolling up  → element rotates back
 * - Scroll stops  → element stops (with a smooth spring ease-out)
 */
export function ScrollRotate({
  children,
  speed = 0.1,
  stiffness = 80,
  damping = 20,
  className,
}: ScrollRotateProps) {
  const { scrollY } = useScroll();

  // Map scroll offset (px) → rotation (deg)
  const rotateRaw = useTransform(scrollY, (latest) => latest * speed);

  // Add a spring so the rotation eases in/out smoothly when scrolling
  // starts and stops, instead of snapping instantly.
  const rotate = useSpring(rotateRaw, { stiffness, damping });

  return (
    <motion.div style={{ rotate }} className={className}>
      {children}
    </motion.div>
  );
}
