/* eslint-disable react-hooks/purity */
/* eslint-disable @typescript-eslint/ban-ts-comment */
"use client";
import { useEffect, useId, useState } from "react";
import { motion } from "framer-motion";
import { cn } from "@/core/lib/utils";

interface SparklesProps {
  id?: string;
  className?: string;
  background?: string;
  minSize?: number;
  maxSize?: number;
  speed?: number;
  particleColor?: string;
  particleDensity?: number;
}

export const SparklesCore = ({
  id,
  className,
  background = "transparent",
  minSize = 0.4,
  maxSize = 1,
  speed = 1,
  particleColor = "#FFF",
  particleDensity = 100,
}: SparklesProps) => {
  const [particles, setParticles] = useState<Array<{
    id: number;
    x: number;
    y: number;
    size: number;
    duration: number;
    delay: number;
  }>>([]);

  const generatedId = useId();
  const uniqueId = id || generatedId;

  useEffect(() => {
    const generateParticles = () => {
      const newParticles = [];
      for (let i = 0; i < particleDensity; i++) {
        // @ts-ignore
        newParticles.push({
          id: i,
          x: Math.random() * 100,
          y: Math.random() * 100,
          size: Math.random() * (maxSize - minSize) + minSize,
          duration: (Math.random() * 2 + 1) / speed,
          delay: Math.random() * 2,
        });
      }
      setParticles(newParticles);
    };

    generateParticles();
  }, [particleDensity, minSize, maxSize, speed]);

  return (
    <div
      className={cn("relative h-full w-full", className)}
      style={{ background }}
    >
      <svg
        className="absolute inset-0 h-full w-full"
        xmlns="http://www.w3.org/2000/svg"
      >
        <defs>
          <filter id={`glow-${uniqueId}`}>
            <feGaussianBlur stdDeviation="1.5" result="coloredBlur" />
            <feMerge>
              <feMergeNode in="coloredBlur" />
              <feMergeNode in="SourceGraphic" />
            </feMerge>
          </filter>
        </defs>
        {particles.map((particle) => (
          <motion.circle
            key={particle.id}
            cx={`${particle.x}%`}
            cy={`${particle.y}%`}
            r={particle.size}
            fill={particleColor}
            filter={`url(#glow-${uniqueId})`}
            initial={{ opacity: 0, scale: 0 }}
            animate={{
              opacity: [0, 1, 1, 0],
              scale: [0, 1, 1, 0],
              y: [0, -10, -20, -30],
            }}
            transition={{
              duration: particle.duration,
              repeat: Infinity,
              delay: particle.delay,
              ease: "easeInOut",
            }}
          />
        ))}
      </svg>
    </div>
  );
};

// Sparkles wrapper component with gradient effect
export const Sparkles = ({
  children,
  className,
  sparklesClassName,
  ...props
}: SparklesProps & {
  children?: React.ReactNode;
  sparklesClassName?: string;
}) => {
  return (
    <div className={cn("relative", className)}>
      {children}
      <div className={cn("absolute inset-0 -z-10", sparklesClassName)}>
        <SparklesCore {...props} />
      </div>
    </div>
  );
};

// Alternative: Canvas-based sparkles for better performance
export const SparklesCanvas = ({
  className,
  background = "transparent",
  minSize = 0.6,
  maxSize = 1.4,
  particleDensity = 50,
  particleColor = "#D4AF37",
  speed = 1,
}: SparklesProps) => {

  const [particles, setParticles] = useState<
    Array<{
      id: number;
      x: number;
      y: number;
      size: number;
      duration: number;
      delay: number;
    }>
  >([]);

  useEffect(() => {
    const newParticles = [];

    for (let i = 0; i < particleDensity; i++) {
      // @ts-ignore
      newParticles.push({
        id: i,
        x: Math.random() * 100,
        y: Math.random() * 100,
        size: Math.random() * (maxSize - minSize) + minSize,
        duration: (Math.random() * 3 + 2) / speed,
        delay: Math.random() * 3,
      });
    }

    setParticles(newParticles);
  }, [particleDensity, minSize, maxSize, speed]);

  return (
    <div
      className={cn("relative w-full overflow-hidden", className)}
      style={{ background, height: "150px" }}
    >
      {/* Gradient overlay */}
      <div className="absolute inset-0 bg-linear-to-t from-transparent via-transparent to-slate-850/50 z-10" />

      {/* Sparkle particles */}
      {particles.map((particle) => (
        <motion.div
          key={particle.id}
          className="absolute rounded-full"
          style={{
            left: `${particle.x}%`,
            top: `${particle.y}%`,
            width: `${particle.size * 4}px`,
            height: `${particle.size * 4}px`,
            background: `radial-gradient(circle, ${particleColor} 0%, transparent 70%)`,
            boxShadow: `0 0 ${particle.size * 6}px ${particleColor}`,
          }}
          initial={{ opacity: 0, scale: 0 }}
          animate={{
            opacity: [0, 1, 1, 0],
            scale: [0, 1, 1.2, 0],
            y: [0, -20, -40, -60],
          }}
          transition={{
            duration: particle.duration,
            repeat: Infinity,
            delay: particle.delay,
            ease: "easeOut",
          }}
        />
      ))}

      {/* Center glow effect */}
      <motion.div
        className="absolute left-1/2 top-0 -translate-x-1/2 w-150 h-25"
        style={{
          background: `radial-gradient(ellipse at center, ${particleColor}20 0%, transparent 70%)`,
        }}
        animate={{
          opacity: [0.3, 0.6, 0.3],
          scale: [1, 1.1, 1],
        }}
        transition={{
          duration: 3,
          repeat: Infinity,
          ease: "easeInOut",
        }}
      />
    </div>
  );
};
