"use client";

import { useEffect, useRef, useCallback } from "react";

interface SmoothScrollProps {
  children: React.ReactNode;
  /** Lerp factor: 0.0 (ultra slow) → 1.0 (instant). Default: 0.09 */
  lerp?: number;
  /** Scroll multiplier for wheel events. Default: 1.0 */
  multiplier?: number;
  /** Enable/disable smooth scroll. Default: true */
  enabled?: boolean;
}

/**
 * SmoothScroll — A dependency-free smooth scroll wrapper for Next.js / React.
 *
 * How it works:
 *  - Locks native overflow and sets the wrapper to a fixed full-screen container.
 *  - Tracks a "target" scroll value updated by wheel/touch/keyboard input.
 *  - Each rAF frame, linearly interpolates the "current" value toward "target"
 *    and applies it as a CSS transform on the inner scroll container.
 *  - This produces the characteristic momentum/inertia feel with zero dependencies.
 *
 * Usage (app/layout.tsx or any page):
 *  <SmoothScroll lerp={0.09} multiplier={1}>
 *    {children}
 *  </SmoothScroll>
 */
export default function SmoothScroll({
  children,
  lerp = 0.09,
  multiplier = 1,
  enabled = true,
}: SmoothScrollProps) {
  const containerRef = useRef<HTMLDivElement>(null);
  const contentRef = useRef<HTMLDivElement>(null);

  const state = useRef({
    current: 0,
    target: 0,
    raf: 0,
    touchStartY: 0,
    lastTouch: 0,
    velocity: 0,
  });

  const clamp = (val: number, min: number, max: number) =>
    Math.min(Math.max(val, min), max);

  const getMaxScroll = useCallback(() => {
    if (!contentRef.current) return 0;
    return contentRef.current.scrollHeight - window.innerHeight;
  }, []);

  const lerp_ = (a: number, b: number, t: number) => a + (b - a) * t;

  const tick = useCallback(() => {
    const s = state.current;
    const max = getMaxScroll();

    s.target = clamp(s.target, 0, max);
    s.current = lerp_(s.current, s.target, lerp);

    // Stop rAF when close enough (avoids wasted frames)
    if (Math.abs(s.target - s.current) < 0.05) {
      s.current = s.target;
    }

    if (contentRef.current) {
      contentRef.current.style.transform = `translate3d(0, ${-s.current}px, 0)`;
    }

    // Sync window.scrollY so anchor links, scroll-based hooks, etc. work
    window.history.scrollRestoration = "manual";

    s.raf = requestAnimationFrame(tick);
  }, [lerp, getMaxScroll]);

  // ── Wheel ────────────────────────────────────────────────────────────────
  const onWheel = useCallback(
    (e: WheelEvent) => {
      e.preventDefault();
      state.current.target += e.deltaY * multiplier;
    },
    [multiplier]
  );

  // ── Touch ────────────────────────────────────────────────────────────────
  const onTouchStart = useCallback((e: TouchEvent) => {
    state.current.touchStartY = e.touches[0].clientY;
    state.current.lastTouch = e.touches[0].clientY;
    state.current.velocity = 0;
  }, []);

  const onTouchMove = useCallback(
    (e: TouchEvent) => {
      e.preventDefault();
      const y = e.touches[0].clientY;
      const delta = state.current.lastTouch - y;
      state.current.velocity = delta;
      state.current.target += delta * multiplier;
      state.current.lastTouch = y;
    },
    [multiplier]
  );

  const onTouchEnd = useCallback(() => {
    // Throw momentum on touch end
    state.current.target += state.current.velocity * 8 * multiplier;
  }, [multiplier]);

  // ── Keyboard ─────────────────────────────────────────────────────────────
  const onKeyDown = useCallback((e: KeyboardEvent) => {
    const step = 120;
    const page = window.innerHeight * 0.85;
    const map: Record<string, number> = {
      ArrowDown: step,
      ArrowUp: -step,
      PageDown: page,
      PageUp: -page,
      End: getMaxScroll(),
      Home: -getMaxScroll(),
      " ": page,
    };
    if (e.key in map) {
      e.preventDefault();
      state.current.target += map[e.key];
    }
  }, [getMaxScroll]);

  // ── Resize ───────────────────────────────────────────────────────────────
  const onResize = useCallback(() => {
    state.current.target = clamp(state.current.target, 0, getMaxScroll());
  }, [getMaxScroll]);

  // ── Mount / Unmount ──────────────────────────────────────────────────────
  useEffect(() => {
    if (!enabled) return;

    const el = containerRef.current;
    if (!el) return;

    // Lock native scroll
    document.documentElement.style.overflow = "hidden";
    document.body.style.overflow = "hidden";

    el.addEventListener("wheel", onWheel, { passive: false });
    el.addEventListener("touchstart", onTouchStart, { passive: true });
    el.addEventListener("touchmove", onTouchMove, { passive: false });
    el.addEventListener("touchend", onTouchEnd, { passive: true });
    window.addEventListener("keydown", onKeyDown);
    window.addEventListener("resize", onResize);

    state.current.raf = requestAnimationFrame(tick);

    return () => {
      document.documentElement.style.overflow = "";
      document.body.style.overflow = "";

      el.removeEventListener("wheel", onWheel);
      el.removeEventListener("touchstart", onTouchStart);
      el.removeEventListener("touchmove", onTouchMove);
      el.removeEventListener("touchend", onTouchEnd);
      window.removeEventListener("keydown", onKeyDown);
      window.removeEventListener("resize", onResize);

      cancelAnimationFrame(state.current.raf);
    };
  }, [enabled, tick, onWheel, onTouchStart, onTouchMove, onTouchEnd, onKeyDown, onResize]);

  // Passthrough when disabled
  if (!enabled) return <>{children}</>;

  return (
    <div
      ref={containerRef}
      style={{
        position: "fixed",
        top: 0,
        left: 0,
        width: "100%",
        height: "100vh",
        overflow: "hidden",
      }}
    >
      <div
        ref={contentRef}
        style={{
          willChange: "transform",
        }}
      >
        {children}
      </div>
    </div>
  );
}
