"use client";

import { useEffect, useRef } from "react";

interface UseSmoothScrollOptions {
  /** Lerp factor: 0.0 (ultra slow) → 1.0 (instant). Default: 0.09 */
  lerp?: number;
  /** Scroll multiplier. Default: 1.0 */
  multiplier?: number;
}

interface SmoothScrollControls {
  /** Scroll to a specific Y position (smoothly) */
  scrollTo: (y: number) => void;
  /** Scroll to an element (smoothly) */
  scrollToElement: (el: HTMLElement, offset?: number) => void;
  /** Get current scroll position */
  getScrollY: () => number;
}

/**
 * useSmoothScroll — hook to programmatically control a <SmoothScroll> instance.
 *
 * Since SmoothScroll intercepts native scroll, you can't use window.scrollTo().
 * Use this hook instead to drive scrolls imperatively (e.g. from nav links).
 *
 * Example:
 *   const { scrollToElement } = useSmoothScroll();
 *   <button onClick={() => scrollToElement(document.getElementById("contact")!)}>
 *     Contact
 *   </button>
 */
export function useSmoothScroll(_options?: UseSmoothScrollOptions): SmoothScrollControls {
  // We communicate via a shared singleton stored on window
  // (SmoothScroll writes its state ref there)
  const scrollTo = (y: number) => {
    const w = window as any;
    if (w.__smoothScrollState) {
      w.__smoothScrollState.target = y;
    } else {
      window.scrollTo({ top: y, behavior: "smooth" });
    }
  };

  const scrollToElement = (el: HTMLElement, offset = 0) => {
    const rect = el.getBoundingClientRect();
    const w = window as any;
    const current = w.__smoothScrollState?.current ?? window.scrollY;
    scrollTo(current + rect.top - offset);
  };

  const getScrollY = (): number => {
    const w = window as any;
    return w.__smoothScrollState?.current ?? window.scrollY;
  };

  return { scrollTo, scrollToElement, getScrollY };
}
