/**
 * Strapi API client (apps/backend)
 * Server-side only. Public API, no auth.
 */

import qs from "qs";
import type {
  StrapiProject,
  StrapiPartner,
  StrapiService,
  StrapiSetting,
  StrapiHome,
  StrapiBlog,
  StrapiTag,
  StrapiCollectionResponse,
  StrapiSingleResponse,
} from "./types";

const STRAPI_URL = process.env.NEXT_PUBLIC_STRAPI_URL ?? process.env.STRAPI_URL ?? "http://localhost:1337";

type FetchOptions = { revalidate?: number | false; tags?: string[]; locale?: string };

async function fetchApi<T>(path: string, options: FetchOptions = {}): Promise<T> {
  const { revalidate = 60, tags } = options;
  const url = `${STRAPI_URL}/api${path}`;

  console.log("full url", url)

  const res = await fetch(url, {
    headers: { "Content-Type": "application/json" },
    next: { tags },
    cache: "no-store",
  });

  if (!res.ok && res.status === 400 && path.includes("?")) {
    const pathWithoutQuery = path.slice(0, path.indexOf("?"));
    const fallbackUrl = `${STRAPI_URL}/api${pathWithoutQuery}`;
    const fallbackRes = await fetch(fallbackUrl, {
      headers: { "Content-Type": "application/json" },
      next: { tags },
    });
    if (fallbackRes.ok) return fallbackRes.json() as Promise<T>;
  }

  if (!res.ok) {
    const body = await res.text();
    console.error(`Strapi API error: ${res.status} ${path}`, body.slice(0, 200));
    throw new Error(`Strapi API: ${res.statusText}`);
  }

  return res.json() as Promise<T>;
}

/** Build full URL for Strapi media (handles relative paths) */
export function getStrapiMediaUrl(url: string | null | undefined): string {
  if (!url) return "";
  if (url.startsWith("http://") || url.startsWith("https://")) return url;
  return `${STRAPI_URL}${url}`;
}

/** Build Strapi v5 REST query string using qs (same lib Strapi uses) */
function buildQuery(params: Record<string, unknown>): string {
  return qs.stringify(params, { encodeValuesOnly: true });
}

// ============ Projects ============

export async function getProjects(options: FetchOptions = {}): Promise<StrapiProject[]> {
  try {
    const query = buildQuery({ populate: "*", sort: "createdAt:desc", locale: options.locale });
    const res = await fetchApi<StrapiCollectionResponse<StrapiProject>>(
      `/projects?${query}`,
      { ...options, tags: ["projects"] }
    );
    return res.data ?? [];
  } catch (e) {
    console.error("getProjects", e);
    return [];
  }
}

export async function getFeaturedProjects(limit: number = 6, options: FetchOptions = {}): Promise<StrapiProject[]> {
  try {
    const query = buildQuery({
      populate: "*",
      sort: "createdAt:desc",
      pagination: { pageSize: limit },
      locale: options.locale,
    });
    const res = await fetchApi<StrapiCollectionResponse<StrapiProject>>(
      `/projects?${query}`,
      { ...options, tags: ["projects"] }
    );
    return res.data ?? [];
  } catch (e) {
    console.error("getFeaturedProjects", e);
    return [];
  }
}

export async function getProjectBySlug(slug: string, options: FetchOptions = {}): Promise<StrapiProject | null> {
  try {
    const query = buildQuery({
      populate: {
        cover_image: { fields: ["url", "alternativeText", "width", "height"] },
        gallery: { fields: ["url", "alternativeText", "width", "height", "name"] },
        service: { populate: { image: { fields: ["url"] } } },
        technologies: { populate: { logo: { fields: ["url"] } } },
        categories: { fields: ["name", "slug"] },
        seo: { populate: { metaImage: { fields: ["url", "alternativeText"] }, openGraph: { populate: { ogImage: { fields: ["url", "alternativeText"] } } } } },
        openGraph: { populate: { ogImage: { fields: ["url", "alternativeText"] } } },
      },
      locale: options.locale,
    });
    const res = await fetchApi<StrapiCollectionResponse<StrapiProject>>(
      `/projects?filters[slug][$eq]=${slug}&${query}`,
      { ...options, tags: ["projects"] }
    );
    return res.data?.[0] ?? null;
  } catch (e) {
    console.error("getProjectBySlug", e);
    return null;
  }
}

/** Optional: get homepage and use its projects relation for curated list */
export async function getHomeProjects(options: FetchOptions = {}): Promise<StrapiProject[]> {
  try {
    const query = buildQuery({ populate: "*" });
    const res = await fetchApi<StrapiSingleResponse<StrapiHome>>(
      `/home?${query}`,
      { ...options, tags: ["home"] }
    );
    return res.data?.projects ?? [];
  } catch (e) {
    console.error("getHomeProjects", e);
    return [];
  }
}

// ============ Services ============

export async function getServices(options: FetchOptions = {}): Promise<StrapiService[]> {
  try {
    const query = buildQuery({
      populate: {
        image: { fields: ["url", "alternativeText", "width", "height"] },
        seo: { populate: { metaImage: { fields: ["url", "alternativeText"] }, openGraph: { populate: { ogImage: { fields: ["url", "alternativeText"] } } } } },
        openGraph: { populate: { ogImage: { fields: ["url", "alternativeText"] } } },
      },
      sort: "createdAt:asc",
    });
    const res = await fetchApi<StrapiCollectionResponse<StrapiService>>(
      `/services?${query}`,
      { ...options, tags: ["services"] }
    );
    return res.data ?? [];
  } catch (e) {
    console.error("getServices", e);
    return [];
  }
}

// ============ Partners ============

export async function getPartners(options: FetchOptions = {}): Promise<StrapiPartner[]> {
  try {
    const query = buildQuery({ populate: "*", sort: "createdAt:asc" });
    const res = await fetchApi<StrapiCollectionResponse<StrapiPartner>>(
      `/partners?${query}`,
      { ...options, tags: ["partners"] }
    );
    return res.data ?? [];
  } catch (e) {
    console.error("getPartners", e);
    return [];
  }
}

// ============ Settings (contact info, social links) ============

/** Map Strapi setting contact_info to simple { email, phone, address, hours } */
export function mapSettingToContactInfo(
  setting: StrapiSetting | null
): { email?: string; phone?: string; address?: string; hours?: string } | null {
  const list = setting?.contact_info;
  if (!list?.length) return null;
  const out: Record<string, string> = {};
  for (const item of list) {
    const v = item.value?.trim();
    if (!v) continue;
    const m = (item.method ?? "").toLowerCase();
    if (m === "email") out.email = v;
    else if (m === "phone") out.phone = v;
    else if (m === "address") out.address = v;
    else if (m === "telegram" || m === "whatsapp") out.phone = out.phone || v;
    else out.hours = out.hours || v;
  }
  return Object.keys(out).length ? out : null;
}

export async function getSettings(options: FetchOptions = {}): Promise<StrapiSetting | null> {
  try {
    const query = buildQuery({
      populate: {
        logo: { fields: ["url", "alternativeText", "width", "height"] },
        social_links: true,
        contact_info: true,
        seo: { populate: { metaImage: { fields: ["url", "alternativeText"] }, openGraph: { populate: { ogImage: { fields: ["url", "alternativeText"] } } } } },
        openGraph: { populate: { ogImage: { fields: ["url", "alternativeText"] } } },
      },
    });
    const res = await fetchApi<StrapiSingleResponse<StrapiSetting>>(
      `/setting?${query}`,
      { ...options, tags: ["setting"] }
    );
    return res.data ?? null;
  } catch (e) {
    console.error("getSettings", e);
    return null;
  }
}

// ============ Blogs ============

export async function getBlogs(
  params: { page?: number; pageSize?: number; tag?: string } = {},
  options: FetchOptions = {}
): Promise<{ data: StrapiBlog[]; meta: { pagination?: { page: number; pageSize: number; pageCount: number; total: number } } }> {
  try {
    const { page = 1, pageSize = 9, tag } = params;
    const queryParams: Record<string, unknown> = {
      populate: {
        cover: { fields: ["url", "alternativeText", "width", "height"] },
        tags: { fields: ["title", "slug"] },
        category: { fields: ["name", "slug"] },
        author: { fields: ["username", "email"] },
      },
      sort: "publishedAt:desc",
      pagination: { page, pageSize },
      locale: options.locale,
    };
    if (tag) {
      queryParams.filters = { tags: { slug: { $eq: tag } } };
    }
    const query = buildQuery(queryParams);
    const res = await fetchApi<StrapiCollectionResponse<StrapiBlog>>(
      `/blogs?${query}`,
      { ...options, tags: ["blogs"] }
    );
    return { data: res.data ?? [], meta: res.meta };
  } catch (e) {
    console.error("getBlogs", e);
    return { data: [], meta: {} };
  }
}

export async function getBlogBySlug(slug: string, options: FetchOptions = {}): Promise<StrapiBlog | null> {
  try {
    const query = buildQuery({
      populate: {
        cover: { fields: ["url", "alternativeText", "width", "height"] },
        tags: { fields: ["title", "slug"] },
        category: { fields: ["name", "slug"] },
        author: { fields: ["username", "email"] },
        seo: { populate: { metaImage: { fields: ["url", "alternativeText"] }, openGraph: { populate: { ogImage: { fields: ["url", "alternativeText"] } } } } },
        openGraph: { populate: { ogImage: { fields: ["url", "alternativeText"] } } },
      },
      locale: options.locale,
    });
    const res = await fetchApi<StrapiCollectionResponse<StrapiBlog>>(
      `/blogs?filters[slug][$eq]=${slug}&${query}`,
      { ...options, tags: ["blogs"] }
    );
    return res.data?.[0] ?? null;
  } catch (e) {
    console.error("getBlogBySlug", e);
    return null;
  }
}

export async function getRecentBlogs(limit: number = 5, options: FetchOptions = {}): Promise<StrapiBlog[]> {
  try {
    const query = buildQuery({
      populate: {
        cover: { fields: ["url", "alternativeText", "width", "height"] },
        tags: { fields: ["title", "slug"] },
        seo: { populate: { metaImage: { fields: ["url", "alternativeText"] }, openGraph: { populate: { ogImage: { fields: ["url", "alternativeText"] } } } } },
        openGraph: { populate: { ogImage: { fields: ["url", "alternativeText"] } } },
      },
      sort: "publishedAt:desc",
      pagination: { pageSize: limit },
      locale: options.locale,
    });
    const res = await fetchApi<StrapiCollectionResponse<StrapiBlog>>(
      `/blogs?${query}`,
      { ...options, tags: ["blogs"] }
    );
    return res.data ?? [];
  } catch (e) {
    console.error("getRecentBlogs", e);
    return [];
  }
}

// ============ Tags ============

export async function getTags(options: FetchOptions = {}): Promise<StrapiTag[]> {
  try {
    const query = buildQuery({ populate: "*", sort: "title:asc" });
    const res = await fetchApi<StrapiCollectionResponse<StrapiTag>>(
      `/tags?${query}`,
      { ...options, tags: ["tags"] }
    );
    return res.data ?? [];
  } catch (e) {
    console.error("getTags", e);
    return [];
  }
}

// ============ Contact form submission (call from server action) ============

export async function submitContact(data: {
  full_name: string;
  email: string;
  phone_number?: string;
  company_name?: string;
  message?: string;
  service?: string; // documentId for relation
}): Promise<{ ok: boolean; error?: string }> {
  try {
    const payload: Record<string, unknown> = {
      full_name: data.full_name,
      email: data.email,
      phone_number: data.phone_number ?? null,
      company_name: data.company_name ?? null,
      message: data.message ?? null,
    };
    if (data.service) {
      payload.service = data.service; // Strapi v5 may accept documentId for relation
    }
    const res = await fetch(`${STRAPI_URL}/api/contact-submissions`, {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ data: payload }),
    });

    if (!res.ok) {
      const text = await res.text();
      return { ok: false, error: text || res.statusText };
    }
    return { ok: true };
  } catch (e) {
    const message = e instanceof Error ? e.message : "Request failed";
    return { ok: false, error: message };
  }
}
