import type { Metadata } from "next";
import { getStrapiMediaUrl } from "./api";
import en from "@/locales/en.json";
import ar from "@/locales/ar.json";

// Minimal shared types for Strapi SEO components used on the frontend
export interface StrapiOpenGraph {
  ogTitle?: string | null;
  ogDescription?: string | null;
  ogImage?: { url: string; alternativeText?: string | null } | null;
  ogUrl?: string | null;
  ogType?: string | null;
}

export interface StrapiSeo {
  metaTitle?: string | null;
  metaDescription?: string | null;
  metaImage?: { url: string; alternativeText?: string | null } | null;
  openGraph?: StrapiOpenGraph | null;
  keywords?: string | null;
  metaRobots?: string | null;
  metaViewport?: string | null;
  canonicalURL?: string | null;
  structuredData?: unknown;
}

type SupportedLocale = "en" | "ar";

const SITE_URL =
  process.env.NEXT_PUBLIC_SITE_URL ||
  process.env.SITE_URL ||
  "https://www.yslootahtech.com";

type FallbackSeo = {
  title: string;
  description: string;
  url?: string;
};

function getCompanyFallbackSeo(locale: SupportedLocale): FallbackSeo {
  if (locale === "ar") {
    const hero = (ar as any).hero;
    return {
      title: `${hero?.title ?? "YS Lootah Tech"} | حلول رقمية متكاملة`,
      description:
        "YS Lootah Tech شريكك الموثوق في التحول الرقمي من دبي، نقدم حلولاً متكاملة تشمل تطوير البرمجيات، تطبيقات الجوال، الذكاء الاصطناعي، الأمن السيبراني، والاستشارات التقنية لمساعدة المؤسسات على الابتكار والنمو.",
      url: SITE_URL,
    };
  }

  const hero = (en as any).hero;
  return {
    title: `${hero?.title ?? "YS Lootah Tech"} | Digital Transformation & Technology Solutions`,
    description:
      "YS Lootah Tech is a Dubai-based technology partner delivering custom software, mobile apps, AI, cybersecurity, and digital transformation solutions that help organizations innovate, scale, and create exceptional digital experiences.",
    url: SITE_URL,
  };
}

export function buildMetadataFromSeo(params: {
  seo?: StrapiSeo | null;
  openGraph?: StrapiOpenGraph | null;
  locale?: SupportedLocale;
  override?: Partial<Metadata>;
}): Metadata {
  const { seo, openGraph, locale = "en", override } = params;
  const fallback = getCompanyFallbackSeo(locale);

  const metaTitle = seo?.metaTitle || fallback.title;
  const metaDescription = seo?.metaDescription || fallback.description;
  const canonical = seo?.canonicalURL || fallback.url;

  const og = seo?.openGraph || openGraph || null;
  const ogTitle = og?.ogTitle || metaTitle;
  const ogDescription = og?.ogDescription || metaDescription;
  const ogUrl = og?.ogUrl || canonical;
  const ogImage = (seo?.metaImage || og?.ogImage) ?? null;

  const metadata: Metadata = {
    title: metaTitle,
    description: metaDescription,
    alternates: canonical
      ? {
          canonical,
        }
      : undefined,
    openGraph: {
      title: ogTitle,
      description: ogDescription,
      url: ogUrl,
      // Cast to any so we can accept arbitrary Strapi values while defaulting to "website"
      type: (og?.ogType as any) || "website",
      images: ogImage
        ? [
            {
              url: getStrapiMediaUrl(ogImage.url),
              alt: ogImage.alternativeText || metaTitle,
            },
          ]
        : undefined,
    },
  };

  if (seo?.metaViewport) {
    metadata.viewport = seo.metaViewport;
  }

  if (seo?.metaRobots) {
    metadata.robots = seo.metaRobots;
  }

  if (seo?.keywords) {
    const keywords = seo.keywords
      .split(",")
      .map((k) => k.trim())
      .filter(Boolean);
    if (keywords.length) {
      metadata.keywords = keywords;
    }
  }

  if (seo?.structuredData) {
    metadata.other = {
      ...(metadata.other || {}),
      structuredData: JSON.stringify(seo.structuredData),
    } as unknown as Metadata["other"];
  }

  return {
    ...metadata,
    ...(override || {}),
  };
}

