import Image from "next/image";
import { getStrapiMediaUrl } from "@/core/lib/api";
import type {
  StrapiBlockNode,
  StrapiInlineNode,
  StrapiTextChild,
} from "@/core/lib/api/types";
import { cn } from "@/core/lib/utils";

/** Render inline text with formatting marks */
function InlineText({ node }: { node: StrapiTextChild }) {
  let el: React.ReactNode = node.text;

  if (node.bold) el = <strong className="font-semibold text-white">{el}</strong>;
  if (node.italic) el = <em>{el}</em>;
  if (node.underline) el = <u className="underline underline-offset-4">{el}</u>;
  if (node.strikethrough) el = <s>{el}</s>;
  if (node.code)
    el = (
      <code className="rounded bg-slate-800 px-1.5 py-0.5 text-sm font-mono text-cyan-400">
        {el}
      </code>
    );

  return <>{el}</>;
}

/** Render a single inline node (text or link) */
function InlineNode({ node }: { node: StrapiInlineNode }) {
  if (node.type === "link") {
    return (
      <a
        href={node.url}
        target="_blank"
        rel="noopener noreferrer"
        className="text-cyan-400 underline underline-offset-4 decoration-cyan-400/40 hover:decoration-cyan-400 transition-colors"
      >
        {node.children.map((child, i) => (
          <InlineText key={i} node={child} />
        ))}
      </a>
    );
  }
  return <InlineText node={node} />;
}

/** Render children array */
function InlineChildren({ children }: { children: StrapiInlineNode[] }) {
  return (
    <>
      {children.map((child, i) => (
        <InlineNode key={i} node={child} />
      ))}
    </>
  );
}

/** Render a single block */
function Block({ block }: { block: StrapiBlockNode }) {
  switch (block.type) {
    case "paragraph":
      return (
        <p className="text-base leading-relaxed text-slate-300 md:text-lg">
          <InlineChildren children={block.children} />
        </p>
      );

    case "heading": {
      const Tag = `h${block.level}` as keyof JSX.IntrinsicElements;
      const sizes: Record<number, string> = {
        1: "text-3xl md:text-4xl font-bold",
        2: "text-2xl md:text-3xl font-bold",
        3: "text-xl md:text-2xl font-semibold",
        4: "text-lg md:text-xl font-semibold",
        5: "text-base md:text-lg font-medium",
        6: "text-sm md:text-base font-medium",
      };
      return (
        <Tag className={cn("text-white mt-8 mb-4", sizes[block.level])}>
          <InlineChildren children={block.children} />
        </Tag>
      );
    }

    case "list": {
      const ListTag = block.format === "ordered" ? "ol" : "ul";
      return (
        <ListTag
          className={cn(
            "space-y-2 pl-6 text-slate-300",
            block.format === "ordered" ? "list-decimal" : "list-disc"
          )}
        >
          {block.children.map((item, i) => (
            <li key={i} className="text-base leading-relaxed md:text-lg">
              <InlineChildren children={item.children} />
            </li>
          ))}
        </ListTag>
      );
    }

    case "image": {
      const url = getStrapiMediaUrl(block.image?.url);
      if (!url) return null;
      return (
        <figure className="my-8">
          <div className="relative aspect-video overflow-hidden rounded-xl border border-slate-800">
            <Image
              src={url}
              alt={block.image?.alternativeText || ""}
              fill
              className="object-cover"
              sizes="(max-width: 768px) 100vw, 720px"
            />
          </div>
          {block.image?.alternativeText && (
            <figcaption className="mt-3 text-center text-sm text-slate-500">
              {block.image.alternativeText}
            </figcaption>
          )}
        </figure>
      );
    }

    case "quote":
      return (
        <blockquote className="my-6 border-l-4 border-cyan-500/50 bg-slate-900/50 py-4 pl-6 pr-4 rounded-r-lg">
          <p className="text-base italic text-slate-300 md:text-lg">
            <InlineChildren children={block.children} />
          </p>
        </blockquote>
      );

    case "code":
      return (
        <pre className="my-6 overflow-x-auto rounded-xl bg-slate-900 border border-slate-800 p-4">
          <code className="text-sm font-mono text-cyan-300">
            {block.children.map((c) => c.text).join("")}
          </code>
        </pre>
      );

    default:
      return null;
  }
}

/** Render an array of Strapi block nodes */
export function BlocksRenderer({ content }: { content: StrapiBlockNode[] }) {
  if (!content?.length) return null;

  return (
    <div className="prose-custom space-y-4">
      {content?.map((block, i) => (
        <Block key={i} block={block} />
      ))}
    </div>
  );
}
