"use client";

import Image from "next/image";
import { Link } from "@/core/lib/i18n/navigation";
import { motion, useMotionValue, useMotionTemplate } from "motion/react";
import { Calendar, Clock, ArrowRight, Tag } from "lucide-react";
import { getStrapiMediaUrl } from "@/core/lib/api";
import type { StrapiBlog } from "@/core/lib/api/types";
import { cn } from "@/core/lib/utils";
import { useCallback } from "react";

// function estimateReadTime(content: StrapiBlog["content"]): string {
//   if (!content?.length) return "3 min read";
//   const text = content
//     .map((block) => {
//       if ("children" in block) {
//         return block.children
//           .map((c: any) => ("text" in c ? c.text : ""))
//           .join(" ");
//       }
//       return "";
//     })
//     .join(" ");
//   const words = text.split(/\s+/).length;
//   const minutes = Math.max(1, Math.ceil(words / 200));
//   return `${minutes} min read`;
// }

function formatDate(dateStr?: string | null): string {
  if (!dateStr) return "";
  return new Date(dateStr).toLocaleDateString("en-US", {
    year: "numeric",
    month: "short",
    day: "numeric",
  });
}

interface BlogCardProps {
  blog: StrapiBlog;
  index?: number;
  featured?: boolean;
}

export function BlogCard({ blog, index = 0, featured = false }: BlogCardProps) {
  const mouseX = useMotionValue(0);
  const mouseY = useMotionValue(0);

  const handleMouseMove = useCallback(
    (event: React.MouseEvent<HTMLDivElement>) => {
      const { currentTarget, clientX, clientY } = event;
      const { left, top } = currentTarget.getBoundingClientRect();
      mouseX.set(clientX - left);
      mouseY.set(clientY - top);
    },
    [mouseX, mouseY]
  );

  const coverUrl = getStrapiMediaUrl(blog.cover?.url);
  const readTime = "3 Min"
  // estimateReadTime(blog.content);

  return (
    <Link href={`/blog/${blog.slug}`}>
      <motion.article
        onMouseMove={handleMouseMove}
        className={cn(
          "group/card relative overflow-hidden rounded-2xl bg-slate-900 border border-slate-800/50",
          featured ? "md:col-span-2 md:row-span-2" : ""
        )}
        initial={{ opacity: 0, y: 40, filter: "blur(15px)" }}
        whileInView={{ opacity: 1, y: 0, filter: "blur(0px)" }}
        viewport={{ once: true, amount: 0.15 }}
        transition={{
          delay: index * 0.08,
          duration: 0.6,
          ease: [0.22, 1, 0.36, 1],
        }}
        whileHover={{ y: -4 }}
      >
        {/* Spotlight Effect */}
        <motion.div
          className="pointer-events-none absolute -inset-px z-20 rounded-[inherit] opacity-0 transition-opacity duration-300 group-hover/card:opacity-100"
          style={{
            background: useMotionTemplate`
              radial-gradient(
                ${featured ? 500 : 350}px circle at ${mouseX}px ${mouseY}px,
                rgba(6, 182, 212, 0.1),
                transparent 80%
              )
            `,
          }}
        />

        {/* Cover Image */}
        <div className={cn("relative overflow-hidden", featured ? "h-64 md:h-80" : "h-48 md:h-56")}>
          {coverUrl ? (
            <Image
              src={coverUrl}
              alt={blog.cover?.alternativeText || blog.title}
              fill
              className="object-cover transition-transform duration-700 group-hover/card:scale-105"
              sizes={featured ? "(max-width: 768px) 100vw, 66vw" : "(max-width: 768px) 100vw, 33vw"}
            />
          ) : (
            <div className="h-full w-full bg-linear-to-br from-slate-800 to-slate-900" />
          )}
          <div className="absolute inset-0 bg-linear-to-t from-slate-900 via-transparent to-transparent" />

          {/* Category badge */}
          {blog.category?.name && (
            <div className="absolute top-4 left-4 z-10">
              <span className="rounded-full bg-cyan-500/20 px-3 py-1 text-xs font-medium text-cyan-400 backdrop-blur-sm border border-cyan-500/20">
                {blog.category.name}
              </span>
            </div>
          )}
        </div>

        {/* Content */}
        <div className="relative z-10 p-5 md:p-6">
          {/* Tags */}
          {blog.tags && blog.tags.length > 0 && (
            <div className="mb-3 flex flex-wrap gap-2">
              {blog.tags.slice(0, 3).map((tag) => (
                <span
                  key={tag.documentId}
                  className="inline-flex items-center gap-1 rounded-md bg-slate-800/80 px-2 py-0.5 text-xs text-slate-400"
                >
                  <Tag className="h-2.5 w-2.5" />
                  {tag.title}
                </span>
              ))}
            </div>
          )}

          {/* Title */}
          <h3
            className={cn(
              "font-bold text-white transition-colors duration-300 group-hover/card:text-cyan-300 line-clamp-2",
              featured ? "text-xl md:text-2xl" : "text-lg"
            )}
          >
            {blog.title}
          </h3>

          {/* Meta */}
          <div className="mt-3 flex items-center gap-4 text-xs text-slate-500">
            {blog.publishedAt && (
              <span className="flex items-center gap-1.5">
                <Calendar className="h-3.5 w-3.5" />
                {formatDate(blog.publishedAt)}
              </span>
            )}
            <span className="flex items-center gap-1.5">
              <Clock className="h-3.5 w-3.5" />
              {readTime}
            </span>
          </div>

          {/* Read more */}
          <div className="mt-4 flex items-center gap-2 text-sm font-medium text-cyan-400 opacity-0 transition-all duration-300 group-hover/card:opacity-100">
            <span>Read article</span>
            <ArrowRight className="h-4 w-4 transition-transform duration-300 group-hover/card:translate-x-1" />
          </div>
        </div>

        {/* Border glow */}
        <div className="pointer-events-none absolute inset-0 rounded-2xl opacity-0 transition-opacity duration-500 group-hover/card:opacity-100">
          <div className="absolute inset-0 rounded-2xl ring-1 ring-cyan-500/20" />
        </div>
      </motion.article>
    </Link>
  );
}
