"use client";

import { Link } from "@/core/lib/i18n/navigation";
import { motion } from "motion/react";
import { ChevronLeft, ChevronRight } from "lucide-react";
import { cn } from "@/core/lib/utils";

interface BlogPaginationProps {
  currentPage: number;
  pageCount: number;
  tag?: string;
}

export function BlogPagination({ currentPage, pageCount, tag }: BlogPaginationProps) {
  if (pageCount <= 1) return null;

  const buildHref = (page: number) => {
    const params = new URLSearchParams();
    if (page > 1) params.set("page", String(page));
    if (tag) params.set("tag", tag);
    const qs = params.toString();
    return `/blog${qs ? `?${qs}` : ""}`;
  };

  const pages = Array.from({ length: pageCount }, (_, i) => i + 1);

  // Show limited page numbers with ellipsis for large sets
  const visiblePages = pages.filter((p) => {
    if (pageCount <= 7) return true;
    if (p === 1 || p === pageCount) return true;
    if (Math.abs(p - currentPage) <= 1) return true;
    return false;
  });

  return (
    <motion.nav
      className="mt-12 flex items-center justify-center gap-2"
      initial={{ opacity: 0, y: 20 }}
      animate={{ opacity: 1, y: 0 }}
      transition={{ delay: 0.3, duration: 0.5 }}
    >
      {/* Previous */}
      {currentPage > 1 ? (
        <Link
          href={buildHref(currentPage - 1)}
          className="flex items-center gap-1 rounded-lg border border-slate-800 bg-slate-900/50 px-3 py-2 text-sm text-slate-400 transition-colors hover:border-cyan-500/30 hover:text-cyan-400"
        >
          <ChevronLeft className="h-4 w-4" />
          Prev
        </Link>
      ) : (
        <span className="flex items-center gap-1 rounded-lg border border-slate-800/50 bg-slate-900/30 px-3 py-2 text-sm text-slate-600 cursor-not-allowed">
          <ChevronLeft className="h-4 w-4" />
          Prev
        </span>
      )}

      {/* Page numbers */}
      <div className="flex items-center gap-1">
        {visiblePages.map((page, i) => {
          // Check if we need ellipsis before this page
          const prevVisible = visiblePages[i - 1];
          const needsEllipsis = prevVisible && page - prevVisible > 1;

          return (
            <span key={page} className="flex items-center gap-1">
              {needsEllipsis && (
                <span className="px-2 text-slate-600">...</span>
              )}
              <Link
                href={buildHref(page)}
                className={cn(
                  "flex h-9 w-9 items-center justify-center rounded-lg text-sm font-medium transition-all duration-200",
                  page === currentPage
                    ? "bg-cyan-500/20 text-cyan-400 border border-cyan-500/30"
                    : "border border-slate-800/50 text-slate-400 hover:border-cyan-500/20 hover:text-cyan-400"
                )}
              >
                {page}
              </Link>
            </span>
          );
        })}
      </div>

      {/* Next */}
      {currentPage < pageCount ? (
        <Link
          href={buildHref(currentPage + 1)}
          className="flex items-center gap-1 rounded-lg border border-slate-800 bg-slate-900/50 px-3 py-2 text-sm text-slate-400 transition-colors hover:border-cyan-500/30 hover:text-cyan-400"
        >
          Next
          <ChevronRight className="h-4 w-4" />
        </Link>
      ) : (
        <span className="flex items-center gap-1 rounded-lg border border-slate-800/50 bg-slate-900/30 px-3 py-2 text-sm text-slate-600 cursor-not-allowed">
          Next
          <ChevronRight className="h-4 w-4" />
        </span>
      )}
    </motion.nav>
  );
}
