"use client";

import Image from "next/image";
import { Link } from "@/core/lib/i18n/navigation";
import { motion } from "motion/react";
import { Calendar, Tag as TagIcon, TrendingUp } from "lucide-react";
import { getStrapiMediaUrl } from "@/core/lib/api";
import type { StrapiBlog, StrapiTag } from "@/core/lib/api/types";
import { cn } from "@/core/lib/utils";

function formatDate(dateStr?: string | null): string {
  if (!dateStr) return "";
  return new Date(dateStr).toLocaleDateString("en-US", {
    month: "short",
    day: "numeric",
    year: "numeric",
  });
}

// ============ Tags Widget ============

function TagsWidget({
  tags,
  activeTag,
}: {
  tags: StrapiTag[];
  activeTag?: string;
}) {
  if (!tags.length) return null;

  return (
    <motion.div
      className="rounded-2xl border border-slate-800/50 bg-slate-900/50 p-5 backdrop-blur-sm"
      initial={{ opacity: 0, y: 20 }}
      animate={{ opacity: 1, y: 0 }}
      transition={{ delay: 0.1, duration: 0.5 }}
    >
      <h3 className="flex items-center gap-2 text-sm font-semibold uppercase tracking-wider text-slate-400">
        <TagIcon className="h-4 w-4 text-cyan-500" />
        Tags
      </h3>

      <div className="mt-4 flex flex-wrap gap-2">
        <Link href="/blog">
          <span
            className={cn(
              "inline-block cursor-pointer rounded-lg px-3 py-1.5 text-xs font-medium transition-all duration-200",
              !activeTag
                ? "bg-cyan-500/20 text-cyan-400 border border-cyan-500/30"
                : "bg-slate-800/80 text-slate-400 hover:bg-slate-800 hover:text-slate-300 border border-slate-700/50"
            )}
          >
            All
          </span>
        </Link>
        {tags.map((tag) => (
          <Link key={tag.documentId} href={`/blog?tag=${tag.slug}`}>
            <span
              className={cn(
                "inline-block cursor-pointer rounded-lg px-3 py-1.5 text-xs font-medium transition-all duration-200",
                activeTag === tag.slug
                  ? "bg-cyan-500/20 text-cyan-400 border border-cyan-500/30"
                  : "bg-slate-800/80 text-slate-400 hover:bg-slate-800 hover:text-slate-300 border border-slate-700/50"
              )}
            >
              {tag.title}
            </span>
          </Link>
        ))}
      </div>
    </motion.div>
  );
}

// ============ Recent Posts Widget ============

function RecentPostsWidget({ posts }: { posts: StrapiBlog[] }) {
  if (!posts.length) return null;

  return (
    <motion.div
      className="rounded-2xl border border-slate-800/50 bg-slate-900/50 p-5 backdrop-blur-sm"
      initial={{ opacity: 0, y: 20 }}
      animate={{ opacity: 1, y: 0 }}
      transition={{ delay: 0.2, duration: 0.5 }}
    >
      <h3 className="flex items-center gap-2 text-sm font-semibold uppercase tracking-wider text-slate-400">
        <TrendingUp className="h-4 w-4 text-cyan-500" />
        Recent Posts
      </h3>

      <div className="mt-4 space-y-4">
        {posts.map((post, i) => {
          const coverUrl = getStrapiMediaUrl(post.cover?.url);
          return (
            <Link key={post.documentId} href={`/blog/${post.slug}`}>
              <motion.div
                className="group flex gap-3 rounded-xl p-2 -mx-2 transition-colors hover:bg-slate-800/50"
                initial={{ opacity: 0, x: -10 }}
                animate={{ opacity: 1, x: 0 }}
                transition={{ delay: 0.3 + i * 0.08 }}
              >
                {/* Thumbnail */}
                <div className="relative h-16 w-16 shrink-0 overflow-hidden rounded-lg">
                  {coverUrl ? (
                    <Image
                      src={coverUrl}
                      alt={post.cover?.alternativeText || post.title}
                      fill
                      className="object-cover transition-transform duration-300 group-hover:scale-110"
                      sizes="64px"
                    />
                  ) : (
                    <div className="h-full w-full bg-linear-to-br from-slate-800 to-slate-700" />
                  )}
                </div>

                {/* Info */}
                <div className="flex flex-col justify-center min-w-0">
                  <h4 className="text-sm font-medium text-slate-300 line-clamp-2 transition-colors group-hover:text-cyan-400">
                    {post.title}
                  </h4>
                  <span className="mt-1 flex items-center gap-1 text-xs text-slate-500">
                    <Calendar className="h-3 w-3" />
                    {formatDate(post.publishedAt)}
                  </span>
                </div>
              </motion.div>
            </Link>
          );
        })}
      </div>
    </motion.div>
  );
}

// ============ Main Sidebar ============

interface BlogSidebarProps {
  tags: StrapiTag[];
  recentPosts: StrapiBlog[];
  activeTag?: string;
}

export function BlogSidebar({ tags, recentPosts, activeTag }: BlogSidebarProps) {
  return (
    <aside className="space-y-6 lg:sticky lg:top-24">
      <TagsWidget tags={tags} activeTag={activeTag} />
      <RecentPostsWidget posts={recentPosts} />
    </aside>
  );
}
