"use client";

import { useTranslations } from "next-intl";
import { motion } from "motion/react";
import { ArrowRight } from "lucide-react";
import type { StrapiBlog } from "@/core/lib/api/types";
import { BlogCard } from "@/modules/blog/components";
import { Link } from "@/core/lib/i18n/navigation";
import { MotionButton } from "@/components/ui/button";
import { cn } from "@/core/lib/utils";

// Grid position classes for the layout on the landing page
const gridClasses = [
  "md:col-span-2 md:row-span-2", // Featured large
  "md:col-span-1 md:row-span-1",
  "md:col-span-1 md:row-span-1",
  "md:col-span-2 md:row-span-1",
];

// --- Main Component ---

interface BlogsProps {
  apiBlogs?: StrapiBlog[];
}

export function Blogs({ apiBlogs }: BlogsProps) {
  const t = useTranslations();

  const blogs = (apiBlogs ?? []).slice(0, 4);

  return (
    <section id="blogs" className="relative bg-slate-950 py-24 md:py-32 overflow-hidden">
      {/* Background */}
      <div className="absolute inset-0 pointer-events-none">
        <div className="absolute top-1/4 right-0 h-96 w-96 bg-blue-600/10 rounded-full blur-[150px]" />
        <div className="absolute bottom-0 left-1/4 h-80 w-80 bg-purple-600/10 rounded-full blur-[120px]" />
      </div>

      {/* Top border */}
      <div className="absolute inset-x-0 top-0 h-px bg-linear-to-r from-transparent via-slate-600/30 to-transparent" />

      <div className="container relative z-10">
        {/* Header */}
        <motion.div
          className="mx-auto max-w-3xl text-center mb-16"
          initial={{ opacity: 0, y: 30, filter: "blur(15px)" }}
          whileInView={{ opacity: 1, y: 0, filter: "blur(0px)" }}
          viewport={{ once: true }}
          transition={{ duration: 0.8, ease: [0.22, 1, 0.36, 1] }}
        >
          <motion.h2
            className="text-3xl font-bold tracking-tight text-white md:text-4xl lg:text-5xl"
            initial={{ opacity: 0, y: 20, filter: "blur(12px)" }}
            whileInView={{ opacity: 1, y: 0, filter: "blur(0px)" }}
            viewport={{ once: true }}
            transition={{ duration: 0.6, delay: 0.1 }}
          >
            {t("blogs.title")}
          </motion.h2>
          <motion.p
            className="mt-4 text-lg text-slate-400 md:text-xl"
            initial={{ opacity: 0, y: 15, filter: "blur(10px)" }}
            whileInView={{ opacity: 1, y: 0, filter: "blur(0px)" }}
            viewport={{ once: true }}
            transition={{ duration: 0.6, delay: 0.2 }}
          >
            {t("blogs.subtitle")}
          </motion.p>
        </motion.div>

        {/* Blog Grid */}
        <div className="grid gap-4 md:grid-cols-4 md:grid-rows-2">
          {blogs.map((blog, index) => (
            <div
              key={blog.documentId ?? blog.slug ?? index}
              className={cn(gridClasses[index] || "md:col-span-1", "min-h-64")}
            >
              <BlogCard
                blog={blog}
                index={index}
                featured={index === 0}
              />
            </div>
          ))}
        </div>

        {/* View All Button */}
        <motion.div
          className="mt-12 text-center"
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ delay: 0.4, duration: 0.6 }}
        >
          <Link href="/blog">
            <MotionButton
              variant="outline"
              size="lg"
              iconRight={<ArrowRight className="h-4 w-4" />}
            >
              {t("viewAll")}
            </MotionButton>
          </Link>
        </motion.div>
      </div>
    </section>
  );
}
