"use client";

import { useState, useMemo } from "react";
import { useTranslations } from "next-intl";
import { motion } from "motion/react";
import { cn } from "@/core/lib/utils";
import { MotionButton } from "@/components/ui/button";
import {
  Mail,
  Phone,
  MapPin,
  Clock,
  Send,
  CheckCircle2,
  Loader2,
  ArrowRight,
} from "lucide-react";
import { submitContactForm } from "@/app/actions/contact";

// --- Types ---

export interface ContactInfoDisplay {
  email?: string;
  phone?: string;
  address?: string;
  hours?: string;
}

export interface ContactServiceOption {
  name: string;
  documentId: string;
}

type FormData = {
  name: string;
  email: string;
  phone: string;
  company: string;
  service: string; // documentId when using API options
  message: string;
};

type FormStatus = "idle" | "sending" | "success" | "error";

// --- Components ---

function FloatingInput({
  id,
  label,
  placeholder,
  type = "text",
  value,
  onChange,
  required = false,
}: {
  id: string;
  label: string;
  placeholder: string;
  type?: string;
  value: string;
  onChange: (value: string) => void;
  required?: boolean;
}) {
  const [isFocused, setIsFocused] = useState(false);

  return (
    <div className="group relative">
      <label
        htmlFor={id}
        className={cn(
          "absolute left-4 transition-all duration-200 pointer-events-none",
          isFocused || value
            ? "-top-2 text-xs bg-slate-900 px-1 text-blue-400"
            : "top-4 text-sm text-slate-400"
        )}
      >
        {label}
        {required && <span className="text-rose-400 ms-0.5">*</span>}
      </label>
      <input
        id={id}
        type={type}
        value={value}
        onChange={(e) => onChange(e.target.value)}
        onFocus={() => setIsFocused(true)}
        onBlur={() => setIsFocused(false)}
        placeholder={isFocused ? placeholder : ""}
        className="w-full rounded-xl border border-slate-700 bg-slate-900/50 px-4 py-4 text-white placeholder:text-slate-500 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500/50 transition-all duration-200"
        required={required}
      />
      {/* Bottom glow on focus */}
      <div
        className={cn(
          "absolute bottom-0 left-1/2 h-px w-0 -translate-x-1/2 bg-linear-to-r from-transparent via-blue-500 to-transparent transition-all duration-300",
          isFocused && "w-full"
        )}
      />
    </div>
  );
}

function FloatingSelect({
  id,
  label,
  placeholder,
  options,
  value,
  onChange,
}: {
  id: string;
  label: string;
  placeholder: string;
  options: string[];
  value: string;
  onChange: (value: string) => void;
}) {
  const [isFocused, setIsFocused] = useState(false);

  return (
    <div className="group relative">
      <label
        htmlFor={id}
        className={cn(
          "absolute left-4 transition-all duration-200 pointer-events-none z-10",
          isFocused || value
            ? "-top-2 text-xs bg-slate-900 px-1 text-blue-400"
            : "top-4 text-sm text-slate-400"
        )}
      >
        {label}
      </label>
      <select
        id={id}
        value={value}
        onChange={(e) => onChange(e.target.value)}
        onFocus={() => setIsFocused(true)}
        onBlur={() => setIsFocused(false)}
        className="w-full appearance-none rounded-xl border border-slate-700 bg-slate-900/50 px-4 py-4 text-white focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500/50 transition-all duration-200 cursor-pointer"
      >
        <option value="" disabled className="bg-slate-900">
          {placeholder}
        </option>
        {options.map((option) => (
          <option key={option} value={option} className="bg-slate-900">
            {option}
          </option>
        ))}
      </select>
      {/* Custom arrow */}
      <div className="absolute right-4 top-1/2 -translate-y-1/2 pointer-events-none">
        <ArrowRight className="h-4 w-4 rotate-90 text-slate-400" />
      </div>
    </div>
  );
}

function FloatingTextarea({
  id,
  label,
  placeholder,
  value,
  onChange,
  rows = 4,
}: {
  id: string;
  label: string;
  placeholder: string;
  value: string;
  onChange: (value: string) => void;
  rows?: number;
}) {
  const [isFocused, setIsFocused] = useState(false);

  return (
    <div className="group relative">
      <label
        htmlFor={id}
        className={cn(
          "absolute left-4 transition-all duration-200 pointer-events-none",
          isFocused || value
            ? "-top-2 text-xs bg-slate-900 px-1 text-blue-400"
            : "top-4 text-sm text-slate-400"
        )}
      >
        {label}
      </label>
      <textarea
        id={id}
        value={value}
        onChange={(e) => onChange(e.target.value)}
        onFocus={() => setIsFocused(true)}
        onBlur={() => setIsFocused(false)}
        placeholder={isFocused ? placeholder : ""}
        rows={rows}
        className="w-full resize-none rounded-xl border border-slate-700 bg-slate-900/50 px-4 py-4 text-white placeholder:text-slate-500 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500/50 transition-all duration-200"
      />
    </div>
  );
}

function ContactInfo({
  icon: Icon,
  label,
  value,
  delay,
}: {
  icon: typeof Mail;
  label: string;
  value: string;
  delay: number;
}) {
  return (
    <motion.div
      className="group flex items-start gap-4"
      initial={{ opacity: 0, x: -20 }}
      whileInView={{ opacity: 1, x: 0 }}
      viewport={{ once: true }}
      transition={{ delay, duration: 0.4 }}
    >
      <div className="flex h-12 w-12 shrink-0 items-center justify-center rounded-xl bg-blue-500/10 text-blue-400 transition-all duration-300 group-hover:bg-blue-500/20 group-hover:scale-110">
        <Icon className="h-5 w-5" />
      </div>
      <div>
        <div className="text-sm text-slate-400">{label}</div>
        <div className="mt-1 font-medium text-white">{value}</div>
      </div>
    </motion.div>
  );
}

// --- Main Component ---

export interface ContactUsProps {
  contactInfo?: ContactInfoDisplay | null;
  serviceOptions?: ContactServiceOption[];
}

export function ContactUs({ contactInfo: apiContactInfo, serviceOptions: apiServiceOptions = [] }: ContactUsProps = {}) {
  const t = useTranslations("contact");
  const [status, setStatus] = useState<FormStatus>("idle");
  const [formData, setFormData] = useState<FormData>({
    name: "",
    email: "",
    phone: "",
    company: "",
    service: "",
    message: "",
  });

  const serviceOptions = useMemo((): string[] => {
    if (apiServiceOptions.length > 0) return apiServiceOptions.map((o) => o.name);
    const raw = t.raw("form.serviceOptions");
    return Array.isArray(raw) ? (raw as string[]) : [];
  }, [apiServiceOptions, t]);

  const selectedServiceDocumentId = useMemo(() => {
    if (apiServiceOptions.length === 0 || !formData.service) return undefined;
    const byName = apiServiceOptions.find((o) => o.name === formData.service);
    return byName?.documentId;
  }, [apiServiceOptions, formData.service]);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setStatus("sending");
    const result = await submitContactForm({
      full_name: formData.name,
      email: formData.email,
      phone_number: formData.phone || undefined,
      company_name: formData.company || undefined,
      message: formData.message || undefined,
      service_documentId: selectedServiceDocumentId,
    });
    if (result.ok) {
      setStatus("success");
      setFormData({ name: "", email: "", phone: "", company: "", service: "", message: "" });
      setTimeout(() => setStatus("idle"), 3000);
    } else {
      setStatus("error");
      setTimeout(() => setStatus("idle"), 3000);
    }
  };

  const updateField = (field: keyof FormData) => (value: string) => {
    setFormData((prev) => ({ ...prev, [field]: value }));
  };

  return (
    <section id="contact" className="relative bg-slate-950 py-24 md:py-32 overflow-hidden">
      {/* Background */}
      <div className="absolute inset-0 pointer-events-none">
        <div className="absolute top-0 left-1/2 -translate-x-1/2 h-96 w-96 bg-blue-600/10 rounded-full blur-[200px]" />
        <div className="absolute bottom-0 right-1/4 h-80 w-80 bg-purple-600/10 rounded-full blur-[150px]" />
      </div>

      {/* Grid pattern */}
      <div
        className="absolute inset-0 opacity-[0.02] pointer-events-none"
        style={{
          backgroundImage: `linear-gradient(rgba(255,255,255,0.1) 1px, transparent 1px),
            linear-gradient(90deg, rgba(255,255,255,0.1) 1px, transparent 1px)`,
          backgroundSize: "60px 60px",
        }}
      />

      {/* Top border */}
      <div className="absolute inset-x-0 top-0 h-px bg-linear-to-r from-transparent via-slate-600/30 to-transparent" />

      <div className="container relative z-10">
        {/* Header */}
        <motion.div
          className="mx-auto max-w-3xl text-center mb-16"
          initial={{ opacity: 0, y: 30, filter: "blur(15px)" }}
          whileInView={{ opacity: 1, y: 0, filter: "blur(0px)" }}
          viewport={{ once: true }}
          transition={{ duration: 0.8, ease: [0.22, 1, 0.36, 1] }}
        >
          <motion.h2
            className="text-3xl font-bold tracking-tight text-white md:text-4xl lg:text-5xl"
            initial={{ opacity: 0, y: 20, filter: "blur(12px)" }}
            whileInView={{ opacity: 1, y: 0, filter: "blur(0px)" }}
            viewport={{ once: true }}
            transition={{ duration: 0.6, delay: 0.1 }}
          >
            {t("title")}
          </motion.h2>
          <motion.p
            className="mt-4 text-lg text-slate-400 md:text-xl"
            initial={{ opacity: 0, y: 15, filter: "blur(10px)" }}
            whileInView={{ opacity: 1, y: 0, filter: "blur(0px)" }}
            viewport={{ once: true }}
            transition={{ duration: 0.6, delay: 0.2 }}
          >
            {t("subtitle")}
          </motion.p>
        </motion.div>

        <div className="grid gap-12 lg:grid-cols-5">
          {/* Contact Information */}
          <motion.div
            className="lg:col-span-2 space-y-8"
            initial={{ opacity: 0, x: -40, filter: "blur(15px)" }}
            whileInView={{ opacity: 1, x: 0, filter: "blur(0px)" }}
            viewport={{ once: true }}
            transition={{ duration: 0.7, ease: [0.22, 1, 0.36, 1] }}
          >
            <div className="rounded-2xl border border-slate-800 bg-slate-900/50 p-8 backdrop-blur-sm">
              <h3 className="text-xl font-semibold text-white mb-6">
                {t("info.title")}
              </h3>

              <div className="space-y-6">
                <ContactInfo
                  icon={Mail}
                  label="Email"
                  value={apiContactInfo?.email ?? t("info.email")}
                  delay={0.1}
                />
                <ContactInfo
                  icon={Phone}
                  label="Phone"
                  value={apiContactInfo?.phone ?? t("info.phone")}
                  delay={0.2}
                />
                <ContactInfo
                  icon={MapPin}
                  label="Address"
                  value={apiContactInfo?.address ?? t("info.address")}
                  delay={0.3}
                />
                <ContactInfo
                  icon={Clock}
                  label="Hours"
                  value={apiContactInfo?.hours ?? t("info.hours")}
                  delay={0.4}
                />
              </div>

              {/* Social Media Links */}
              <motion.div
                className="mt-8 pt-6 border-t border-slate-800"
                initial={{ opacity: 0, y: 10 }}
                whileInView={{ opacity: 1, y: 0 }}
                viewport={{ once: true }}
                transition={{ delay: 0.5, duration: 0.4 }}
              >
                <div className="text-sm text-slate-400 mb-4">{t("social.title")}</div>
                <div className="flex gap-3">
                  {/* Instagram */}
                  <a
                    href={t("social.instagram")}
                    target="_blank"
                    rel="noopener noreferrer"
                    className="group flex h-11 w-11 items-center justify-center rounded-xl border border-slate-700 bg-slate-800/50 transition-all duration-300 hover:border-pink-500/50 hover:bg-pink-500/10"
                  >
                    <svg
                      className="h-5 w-5 text-slate-400 transition-colors group-hover:text-pink-400"
                      fill="currentColor"
                      viewBox="0 0 24 24"
                    >
                      <path d="M12 2.163c3.204 0 3.584.012 4.85.07 3.252.148 4.771 1.691 4.919 4.919.058 1.265.069 1.645.069 4.849 0 3.205-.012 3.584-.069 4.849-.149 3.225-1.664 4.771-4.919 4.919-1.266.058-1.644.07-4.85.07-3.204 0-3.584-.012-4.849-.07-3.26-.149-4.771-1.699-4.919-4.92-.058-1.265-.07-1.644-.07-4.849 0-3.204.013-3.583.07-4.849.149-3.227 1.664-4.771 4.919-4.919 1.266-.057 1.645-.069 4.849-.069zm0-2.163c-3.259 0-3.667.014-4.947.072-4.358.2-6.78 2.618-6.98 6.98-.059 1.281-.073 1.689-.073 4.948 0 3.259.014 3.668.072 4.948.2 4.358 2.618 6.78 6.98 6.98 1.281.058 1.689.072 4.948.072 3.259 0 3.668-.014 4.948-.072 4.354-.2 6.782-2.618 6.979-6.98.059-1.28.073-1.689.073-4.948 0-3.259-.014-3.667-.072-4.947-.196-4.354-2.617-6.78-6.979-6.98-1.281-.059-1.69-.073-4.949-.073zm0 5.838c-3.403 0-6.162 2.759-6.162 6.162s2.759 6.163 6.162 6.163 6.162-2.759 6.162-6.163c0-3.403-2.759-6.162-6.162-6.162zm0 10.162c-2.209 0-4-1.79-4-4 0-2.209 1.791-4 4-4s4 1.791 4 4c0 2.21-1.791 4-4 4zm6.406-11.845c-.796 0-1.441.645-1.441 1.44s.645 1.44 1.441 1.44c.795 0 1.439-.645 1.439-1.44s-.644-1.44-1.439-1.44z" />
                    </svg>
                  </a>

                  {/* Facebook */}
                  <a
                    href={t("social.facebook")}
                    target="_blank"
                    rel="noopener noreferrer"
                    className="group flex h-11 w-11 items-center justify-center rounded-xl border border-slate-700 bg-slate-800/50 transition-all duration-300 hover:border-blue-500/50 hover:bg-blue-500/10"
                  >
                    <svg
                      className="h-5 w-5 text-slate-400 transition-colors group-hover:text-blue-400"
                      fill="currentColor"
                      viewBox="0 0 24 24"
                    >
                      <path d="M24 12.073c0-6.627-5.373-12-12-12s-12 5.373-12 12c0 5.99 4.388 10.954 10.125 11.854v-8.385H7.078v-3.47h3.047V9.43c0-3.007 1.792-4.669 4.533-4.669 1.312 0 2.686.235 2.686.235v2.953H15.83c-1.491 0-1.956.925-1.956 1.874v2.25h3.328l-.532 3.47h-2.796v8.385C19.612 23.027 24 18.062 24 12.073z" />
                    </svg>
                  </a>

                  {/* LinkedIn */}
                  <a
                    href={t("social.linkedin")}
                    target="_blank"
                    rel="noopener noreferrer"
                    className="group flex h-11 w-11 items-center justify-center rounded-xl border border-slate-700 bg-slate-800/50 transition-all duration-300 hover:border-sky-500/50 hover:bg-sky-500/10"
                  >
                    <svg
                      className="h-5 w-5 text-slate-400 transition-colors group-hover:text-sky-400"
                      fill="currentColor"
                      viewBox="0 0 24 24"
                    >
                      <path d="M20.447 20.452h-3.554v-5.569c0-1.328-.027-3.037-1.852-3.037-1.853 0-2.136 1.445-2.136 2.939v5.667H9.351V9h3.414v1.561h.046c.477-.9 1.637-1.85 3.37-1.85 3.601 0 4.267 2.37 4.267 5.455v6.286zM5.337 7.433c-1.144 0-2.063-.926-2.063-2.065 0-1.138.92-2.063 2.063-2.063 1.14 0 2.064.925 2.064 2.063 0 1.139-.925 2.065-2.064 2.065zm1.782 13.019H3.555V9h3.564v11.452zM22.225 0H1.771C.792 0 0 .774 0 1.729v20.542C0 23.227.792 24 1.771 24h20.451C23.2 24 24 23.227 24 22.271V1.729C24 .774 23.2 0 22.222 0h.003z" />
                    </svg>
                  </a>

                  {/* WhatsApp */}
                  <a
                    href={t("social.whatsapp")}
                    target="_blank"
                    rel="noopener noreferrer"
                    className="group flex h-11 w-11 items-center justify-center rounded-xl border border-slate-700 bg-slate-800/50 transition-all duration-300 hover:border-emerald-500/50 hover:bg-emerald-500/10"
                  >
                    <svg
                      className="h-5 w-5 text-slate-400 transition-colors group-hover:text-emerald-400"
                      fill="currentColor"
                      viewBox="0 0 24 24"
                    >
                      <path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347m-5.421 7.403h-.004a9.87 9.87 0 01-5.031-1.378l-.361-.214-3.741.982.998-3.648-.235-.374a9.86 9.86 0 01-1.51-5.26c.001-5.45 4.436-9.884 9.888-9.884 2.64 0 5.122 1.03 6.988 2.898a9.825 9.825 0 012.893 6.994c-.003 5.45-4.437 9.884-9.885 9.884m8.413-18.297A11.815 11.815 0 0012.05 0C5.495 0 .16 5.335.157 11.892c0 2.096.547 4.142 1.588 5.945L.057 24l6.305-1.654a11.882 11.882 0 005.683 1.448h.005c6.554 0 11.89-5.335 11.893-11.893a11.821 11.821 0 00-3.48-8.413z" />
                    </svg>
                  </a>
                </div>
              </motion.div>
            </div>


          </motion.div>

          {/* Contact Form */}
          <motion.div
            className="lg:col-span-3"
            initial={{ opacity: 0, x: 40, filter: "blur(15px)" }}
            whileInView={{ opacity: 1, x: 0, filter: "blur(0px)" }}
            viewport={{ once: true }}
            transition={{ duration: 0.7, delay: 0.1, ease: [0.22, 1, 0.36, 1] }}
          >
            <form
              onSubmit={handleSubmit}
              className="rounded-2xl border border-slate-800 bg-slate-900/50 p-8 backdrop-blur-sm"
            >
              <div className="grid gap-6 md:grid-cols-2">
                <FloatingInput
                  id="name"
                  label={t("form.name")}
                  placeholder={t("form.namePlaceholder")}
                  value={formData.name}
                  onChange={updateField("name")}
                  required
                />
                <FloatingInput
                  id="email"
                  label={t("form.email")}
                  placeholder={t("form.emailPlaceholder")}
                  type="email"
                  value={formData.email}
                  onChange={updateField("email")}
                  required
                />
                <FloatingInput
                  id="phone"
                  label={t("form.phone")}
                  placeholder={t("form.phonePlaceholder")}
                  type="tel"
                  value={formData.phone}
                  onChange={updateField("phone")}
                />
                <FloatingInput
                  id="company"
                  label={t("form.company")}
                  placeholder={t("form.companyPlaceholder")}
                  value={formData.company}
                  onChange={updateField("company")}
                />
              </div>

              <div className="mt-6">
                <FloatingSelect

                  id="service"
                  label={t("form.service")}
                  placeholder={""}
                  options={serviceOptions}
                  value={formData.service}
                  onChange={updateField("service")}
                />
              </div>

              <div className="mt-6">
                <FloatingTextarea
                  id="message"
                  label={t("form.message")}
                  placeholder={t("form.messagePlaceholder")}
                  value={formData.message}
                  onChange={updateField("message")}
                  rows={5}
                />
              </div>

              {/* Submit Button */}
              <MotionButton
                type="submit"
                variant={status === "success" ? "secondary" : status === "error" ? "secondary" : "gradient"}
                size="lg"
                disabled={status === "sending" || status === "success"}
                className={cn(
                  "mt-8 w-full rounded-xl py-4",
                  status === "success" && "bg-emerald-600 hover:bg-emerald-600",
                  status === "error" && "bg-red-600/90 hover:bg-red-600/90"
                )}
                whileHover={{ scale: status === "idle" ? 1.01 : 1 }}
                whileTap={{ scale: status === "idle" ? 0.99 : 1 }}
                iconLeft={
                  status === "idle" ? (
                    <Send className="h-5 w-5" />
                  ) : status === "sending" ? (
                    <Loader2 className="h-5 w-5 animate-spin" />
                  ) : (
                    <CheckCircle2 className="h-5 w-5" />
                  )
                }
                shine={status === "idle"}
                glow={status === "idle"}
              >
                {status === "idle" && t("form.submit")}
                {status === "sending" && t("form.sending")}
                {status === "success" && t("form.success")}
                {status === "error" && t("form.error")}
              </MotionButton>

              {/* Form decorative glow */}
              <div className="absolute -bottom-20 left-1/2 -translate-x-1/2 h-40 w-80 bg-blue-500/10 rounded-full blur-[100px] pointer-events-none" />
            </form>
          </motion.div>
        </div>
      </div>
    </section>
  );
}
