"use client";

import React, { useRef } from "react";
import {
  motion,
  useScroll,
  useTransform,
  useSpring,
  MotionValue,
} from "motion/react";
import Image from "next/image";
import Link from "next/link";
import { ArrowRight } from "lucide-react";
import { AnimatedButton } from "@/components/ui/aninmated-button";
import type { StrapiProject, StrapiBlockNode } from "@/core/lib/api/types";
import { getStrapiMediaUrl } from "@/core/lib/api";

export interface ProjectCardData {
  title: string;
  link: string;
  thumbnail: string;
  client: string;
  category: string;
  overview: string;
  services: string[];
  slug: string;
}

function getTextFromBlocks(blocks: StrapiBlockNode[]): string {
  return blocks
    .map((block) => {
      if ("children" in block) {
        return block.children
          .map((c) => ("text" in c ? c.text : ""))
          .join(" ");
      }
      return "";
    })
    .join(" ");
}

function buildOverviewExcerpt(overview: StrapiProject["overview"]): string {
  if (!overview) return "";

  let text = "";

  if (typeof overview === "string") {
    text = overview;
  } else if (Array.isArray(overview)) {
    text = getTextFromBlocks(overview as StrapiBlockNode[]);
  }

  // Strip any residual HTML tags and trim
  const plain = text.replace(/<[^>]*>/g, "").trim();
  if (!plain) return "";

  const truncated = plain.slice(0, 200);
  return truncated + (plain.length > 200 ? "..." : "");
}

function strapiProjectToCard(p: StrapiProject): ProjectCardData {
  const overviewText = buildOverviewExcerpt(p.overview);

  return {
    title: p.title,
    link: p.live_url ?? "#",
    thumbnail: p.cover_image?.url ? getStrapiMediaUrl(p.cover_image.url) : "/images/projects/placeholder.png",
    client: "Client",
    category: p.service?.name ?? "Project",
    overview:
      overviewText ||
      "A comprehensive digital solution designed to transform business operations and enhance user engagement.",
    services: (p.technologies ?? []).slice(0, 3).map((t) => t.title ?? t.documentId).filter(Boolean),
    slug: p.slug,
  };
}

const fallbackProjects: ProjectCardData[] = Array.from({ length: 6 }).map((_, i) => ({
  title: `Project ${i + 1}`,
  link: "#",
  thumbnail: "/images/projects/placeholder.png",
  client: `Client ${i + 1}`,
  category: i % 3 === 0 ? "E-Commerce" : i % 3 === 1 ? "ERP System" : "Mobile App",
  overview:
    "A comprehensive digital solution designed to transform business operations and enhance user engagement through cutting-edge technology and intuitive design.",
  services: ["UI/UX Design", "Full Stack Development", "Cloud Infrastructure"],
  slug: `project-${i + 1}`,
}));

function ProjectCard({
  project,
  index,
  progress,
  range,
  targetScale,
}: {
  project: ProjectCardData;
  index: number;
  progress: MotionValue<number>;
  range: [number, number];
  targetScale: number;
}) {
  const containerRef = useRef<HTMLDivElement>(null);

  const scale = useTransform(progress, range, [1, targetScale]);
  const springScale = useSpring(scale, { stiffness: 300, damping: 30 });

  // Parallax for the image
  const imageY = useTransform(
    progress,
    [range[0], range[1]],
    ["0%", "-10%"]
  );

  // Content reveal animation
  const contentOpacity = useTransform(
    progress,
    [range[0], range[0] + 0.05, range[1] - 0.05, range[1]],
    [0, 1, 1, 0.9]
  );

  const contentY = useTransform(
    progress,
    [range[0], range[0] + 0.1],
    [60, 0]
  );
  const springContentY = useSpring(contentY, { stiffness: 200, damping: 25 });

  // Border glow opacity
  const borderOpacity = useTransform(progress, range, [0.2, 0.6]);

  const isEven = index % 2 === 0;

  return (
    <div
      ref={containerRef}
      className="h-screen w-full sticky top-0 flex items-center justify-center"
      style={{ zIndex: index }}
    >
      <motion.div
        style={{ scale: springScale }}
        className="relative w-full h-[90vh] max-w-350 mx-4 md:mx-8 rounded-3xl overflow-hidden bg-slate-900 shadow-2xl shadow-black/50"
      >
        {/* Background gradient overlay */}
        {/* <div className="absolute inset-0 bg-linear-to-br from-slate-900/95 via-slate-900/80 to-transparent z-10" /> */}

        {/* Animated border glow */}
        <motion.div
          className="absolute inset-0 rounded-3xl z-0"
          style={{
            background: `linear-gradient(${90 + index * 30}deg, transparent, var(--primary), transparent)`,
            opacity: borderOpacity,
          }}
        />

        {/* Main grid layout */}
        <div
          className={`relative z-20 h-full grid grid-cols-1 lg:grid-cols-2 gap-0 ${isEven ? "" : "lg:[direction:rtl]"
            }`}
        >
          {/* Image Section */}
          <div className="relative h-[40vh] lg:h-full overflow-hidden [direction:ltr]">
            <motion.div
              style={{ y: imageY }}
              className="absolute inset-0 scale-110"
            >
              <Image
                src={project.thumbnail}
                alt={project.title}
                fill
                className="object-cover object-top-left"
                priority={index < 2}
              />
            </motion.div>

            {/* Image overlay gradient */}
            {/* <div
              className={`absolute inset-0 ${isEven
                ? "bg-linear-to-r from-transparent via-transparent to-slate-900"
                : "bg-linear-to-l from-transparent via-transparent to-slate-900"
                } hidden lg:block`}
            /> */}
            <div className="absolute inset-0 bg-linear-to-t from-slate-900 via-transparent to-transparent lg:hidden" />

            {/* Project number indicator */}
            <motion.div
              style={{ opacity: contentOpacity }}
              className="absolute top-6 left-6 z-30"
            >
              <span className="text-8xl md:text-9xl font-black text-white/10 select-none">
                {String(index + 1).padStart(2, "0")}
              </span>
            </motion.div>
          </div>

          {/* Content Section */}
          <motion.div
            style={{ opacity: contentOpacity, y: springContentY }}
            className="relative flex flex-col justify-center p-8 md:p-12 lg:p-16 [direction:ltr]"
          >
            {/* Category badge */}
            <motion.div
              initial={{ width: 0 }}
              whileInView={{ width: "auto" }}
              transition={{ duration: 0.5, delay: 0.2 }}
              className="overflow-hidden mb-6"
            >
              <span className="inline-flex items-center gap-2 px-4 py-2 bg-blue-500/10 border border-blue-500/30 rounded-full text-blue-400 text-sm font-semibold uppercase tracking-wider">
                <span className="w-2 h-2 bg-blue-400 rounded-full animate-pulse" />
                {project.category}
              </span>
            </motion.div>

            {/* Title */}
            <h3 className="text-4xl md:text-5xl lg:text-6xl font-bold text-white mb-6 leading-tight">
              {project.title}
            </h3>

            {/* Overview */}
            <p className="text-lg md:text-xl text-slate-400 mb-8 leading-relaxed max-w-xl">
              {project.overview}
            </p>

            {/* Services */}
            <div className="flex flex-wrap gap-3 mb-10">
              {project.services.map((service, i) => (
                <motion.span
                  key={service}
                  initial={{ opacity: 0, y: 20 }}
                  whileInView={{ opacity: 1, y: 0 }}
                  transition={{ duration: 0.4, delay: 0.1 * i }}
                  className="px-4 py-2 bg-slate-800/50 border border-slate-700/50 rounded-xl text-slate-300 text-sm font-medium backdrop-blur-sm"
                >
                  {service}
                </motion.span>
              ))}
            </div>

            {/* Client & CTA */}
            <div className="flex flex-col sm:flex-row sm:items-center justify-between gap-6 pt-8 border-t border-slate-800/50">
              <div className="flex items-center gap-4">
                <div className="w-12 h-12 rounded-full bg-linear-to-br from-blue-500 to-purple-600 flex items-center justify-center text-white font-bold text-lg">
                  {project.client.charAt(0)}
                </div>
                <div>
                  <p className="text-slate-500 text-sm">Client</p>
                  <p className="text-white font-semibold">{project.client}</p>
                </div>
              </div>

              <AnimatedButton
                variant="gradient"
                size="lg"
                iconRight={<ArrowRight className="w-5 h-5" />}
                asChild
              >
                <Link href={`/projects/${project.slug}`}>
                  View Project
                </Link>
              </AnimatedButton>
            </div>
          </motion.div>
        </div>

        {/* Decorative elements */}
        <div className="absolute bottom-0 left-0 w-full h-px bg-linear-to-r from-transparent via-blue-500/50 to-transparent" />
        <div className="absolute top-0 right-0 w-px h-full bg-linear-to-b from-transparent via-purple-500/30 to-transparent" />
      </motion.div>
    </div>
  );
}

function ProgressDot({
  progress,
  index,
  total,
}: {
  progress: MotionValue<number>;
  index: number;
  total: number;
}) {
  const backgroundColor = useTransform(
    progress,
    [
      index / total,
      (index + 0.5) / total,
      (index + 1) / total,
    ],
    ["rgb(51, 65, 85)", "rgb(59, 130, 246)", "rgb(51, 65, 85)"]
  );

  const scale = useTransform(
    progress,
    [
      index / total,
      (index + 0.5) / total,
      (index + 1) / total,
    ],
    [1, 1.5, 1]
  );

  return (
    <motion.div
      className="w-2 h-2 rounded-full bg-slate-700"
      style={{ backgroundColor, scale }}
    />
  );
}

function ProgressIndicator({
  progress,
  totalProjects,
}: {
  progress: MotionValue<number>;
  totalProjects: number;
}) {
  return (
    <div className="fixed right-8 top-1/2 -translate-y-1/2 z-50 hidden lg:flex flex-col gap-2">
      {Array.from({ length: totalProjects }).map((_, i) => (
        <ProgressDot key={i} progress={progress} index={i} total={totalProjects} />
      ))}
    </div>
  );
}

export interface FeaturedProjectProps {
  projects: StrapiProject[];
}

export function FeaturedProject({ projects }: FeaturedProjectProps) {
  const containerRef = useRef<HTMLDivElement>(null);
  const { scrollYProgress } = useScroll({
    target: containerRef,
    offset: ["start start", "end end"],
  });

  const projectCards: ProjectCardData[] =
    projects.length > 0 ? projects.map(strapiProjectToCard) : fallbackProjects;

  return (
    <section
      id="featured-project"
      ref={containerRef}
      className="relative bg-slate-950"
    >
      <div className="sticky top-0 z-30 pointer-events-none">
        <div className="absolute top-0 left-0 right-0 h-32 bg-linear-to-b from-slate-950 via-slate-950/80 to-transparent" />
      </div>
      <div className="absolute inset-x-0 top-0 z-10 h-px bg-linear-to-r from-transparent via-blue-500/50 to-transparent" />

      <div className="relative z-20 min-h-screen flex flex-col items-center justify-center px-4 text-center">
        <motion.div
          initial={{ opacity: 0, y: 40 }}
          whileInView={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.8 }}
          className="max-w-4xl"
        >
          <motion.span
            initial={{ opacity: 0, scale: 0.8 }}
            whileInView={{ opacity: 1, scale: 1 }}
            transition={{ duration: 0.5 }}
            className="inline-block px-4 py-2 mb-8 text-sm font-semibold text-blue-400 bg-blue-500/10 border border-blue-500/20 rounded-full"
          >
            Our Portfolio
          </motion.span>
          <h2 className="text-5xl md:text-7xl lg:text-8xl font-bold text-white mb-8">
            Featured{" "}
            <span className="text-transparent bg-clip-text bg-linear-to-r from-blue-400 via-purple-500 to-pink-500">
              Projects
            </span>
          </h2>
          <p className="text-xl md:text-2xl text-slate-400 max-w-2xl mx-auto mb-12">
            Explore our latest work showcasing innovative solutions and creative excellence
          </p>
          <motion.div
            animate={{ y: [0, 10, 0] }}
            transition={{ duration: 2, repeat: Infinity }}
            className="flex flex-col items-center gap-2 text-slate-500"
          >
            <span className="text-sm font-medium">Scroll to explore</span>
            <svg className="w-6 h-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 14l-7 7m0 0l-7-7m7 7V3" />
            </svg>
          </motion.div>
        </motion.div>
        <div className="absolute inset-0 overflow-hidden pointer-events-none">
          <div className="absolute top-1/4 left-1/4 w-96 h-96 bg-blue-500/5 rounded-full blur-3xl" />
          <div className="absolute bottom-1/4 right-1/4 w-96 h-96 bg-purple-500/5 rounded-full blur-3xl" />
        </div>
      </div>

      <div className="relative">
        {projectCards.map((project, i) => {
          const targetScale = 1 - (projectCards.length - i) * 0.03;
          const start = i / projectCards.length;
          const end = (i + 1) / projectCards.length;
          return (
            <ProjectCard
              key={project.slug}
              project={project}
              index={i}
              progress={scrollYProgress}
              range={[start, end]}
              targetScale={targetScale}
            />
          );
        })}
      </div>
      <ProgressIndicator progress={scrollYProgress} totalProjects={projectCards.length} />
      <div className="h-32 bg-linear-to-b from-transparent to-slate-950" />
    </section>
  );
}
