"use client";

import { useTranslations } from "next-intl";
import { CardSpotlight } from "@/components/ui/card-spotlight";
import { motion, type Variants, type Easing } from "motion/react";
import {
  Award,
  Layers,
  HeartHandshake,
  Check,
  type LucideIcon,
} from "lucide-react";

// Custom easing
const customEase: Easing = [0.22, 1, 0.36, 1];

// Animation variants
const containerVariants: Variants = {
  hidden: { opacity: 0 },
  visible: {
    opacity: 1,
    transition: {
      staggerChildren: 0.15,
      delayChildren: 0.2,
    },
  },
};

const headerVariants: Variants = {
  hidden: {
    opacity: 0,
    y: 30,
    filter: "blur(10px)"
  },
  visible: {
    opacity: 1,
    y: 0,
    filter: "blur(0px)",
    transition: {
      duration: 0.8,
      ease: customEase,
    },
  },
};

const cardVariants: Variants = {
  hidden: {
    opacity: 0,
    y: 50,
    filter: "blur(15px)",
    scale: 0.95,
  },
  visible: {
    opacity: 1,
    y: 0,
    filter: "blur(0px)",
    scale: 1,
    transition: {
      duration: 0.7,
      ease: customEase,
    },
  },
};

const itemVariants: Variants = {
  hidden: {
    opacity: 0,
    x: -20,
    filter: "blur(8px)"
  },
  visible: {
    opacity: 1,
    x: 0,
    filter: "blur(0px)",
    transition: {
      duration: 0.5,
      ease: customEase,
    },
  },
};

function FeaturePillar({
  title,
  items,
  icon: Icon,
  index,
}: {
  title: string;
  items: [string, string, string];
  icon: LucideIcon;
  index: number;
}) {
  return (
    <motion.div
      variants={cardVariants}
      whileHover={{
        y: -8,
        scale: 1.02,
        transition: { duration: 0.3 }
      }}
    >
      <CardSpotlight
        radius={400}
        color="rgba(59, 130, 246, 0.12)"
        className="group h-full rounded-xl border border-slate-700/80 bg-slate-900/60 backdrop-blur-sm hover:border-blue-500/50 transition-all duration-300 hover:shadow-[0_0_30px_-8px_rgba(59,130,246,0.4)]"
      >
        <div className="relative z-10 p-6 md:p-8 h-full flex flex-col">
          <motion.div
            className="mb-5 flex h-12 w-12 items-center justify-center rounded-lg bg-blue-500/10 text-blue-400 ring-1 ring-blue-500/20 group-hover/spotlight:ring-blue-400/40 transition-all"
            whileHover={{
              rotate: [0, -10, 10, 0],
              scale: 1.1,
              transition: { duration: 0.5 }
            }}
          >
            <Icon className="h-6 w-6" />
          </motion.div>
          <motion.h3
            className="text-xl font-semibold text-white mb-4 tracking-tight"
            initial={{ opacity: 0 }}
            whileInView={{ opacity: 1 }}
            transition={{ delay: 0.2 + index * 0.1 }}
          >
            {title}
          </motion.h3>
          <motion.ul
            className="space-y-3 flex-1"
            variants={containerVariants}
            initial="hidden"
            whileInView="visible"
            viewport={{ once: true, amount: 0.3 }}
          >
            {items.map((item, i) => (
              <motion.li
                key={i}
                className="flex gap-3 text-sm text-slate-300"
                variants={itemVariants}
                custom={i}
              >
                <motion.span
                  className="mt-0.5 shrink-0 flex h-5 w-5 items-center justify-center rounded-full bg-emerald-500/15 text-emerald-400"
                  whileHover={{ scale: 1.2 }}
                >
                  <Check className="h-3 w-3" strokeWidth={2.5} />
                </motion.span>
                <span>{item}</span>
              </motion.li>
            ))}
          </motion.ul>
        </div>
      </CardSpotlight>
    </motion.div>
  );
}

export function Features() {
  const t = useTranslations("features");

  const expertiseItems: [string, string, string] = [
    t("expertise.item1"),
    t("expertise.item2"),
    t("expertise.item3"),
  ];
  const servicesItems: [string, string, string] = [
    t("services.item1"),
    t("services.item2"),
    t("services.item3"),
  ];
  const commitmentItems: [string, string, string] = [
    t("commitment.item1"),
    t("commitment.item2"),
    t("commitment.item3"),
  ];

  return (
    <section
      id="features"
      className="relative w-full overflow-hidden bg-slate-950 py-24 md:py-32"
    >
      {/* Top edge glow */}
      <motion.div
        className="absolute inset-x-0 top-0 h-px bg-linear-to-r from-transparent via-blue-500/50 to-transparent"
        initial={{ scaleX: 0, opacity: 0 }}
        whileInView={{ scaleX: 1, opacity: 1 }}
        viewport={{ once: true }}
        transition={{ duration: 1, ease: "easeOut" }}
      />

      <div className="container relative z-10">
        <motion.div
          className="mx-auto max-w-3xl text-center"
          initial="hidden"
          whileInView="visible"
          viewport={{ once: true, amount: 0.3 }}
          variants={containerVariants}
        >
          <motion.h2
            className="text-3xl font-bold tracking-tight text-white md:text-4xl lg:text-5xl"
            variants={headerVariants}
          >
            {t("title")}
          </motion.h2>
          <motion.p
            className="mt-4 text-lg text-slate-400 md:text-xl"
            variants={headerVariants}
          >
            {t("subtitle")}
          </motion.p>
        </motion.div>

        <motion.div
          className="mx-auto mt-16 grid max-w-6xl grid-cols-1 gap-6 md:grid-cols-3 md:gap-8"
          variants={containerVariants}
          initial="hidden"
          whileInView="visible"
          viewport={{ once: true, amount: 0.2 }}
        >
          <FeaturePillar
            title={t("expertise.title")}
            items={expertiseItems}
            icon={Award}
            index={0}
          />
          <FeaturePillar
            title={t("services.title")}
            items={servicesItems}
            icon={Layers}
            index={1}
          />
          <FeaturePillar
            title={t("commitment.title")}
            items={commitmentItems}
            icon={HeartHandshake}
            index={2}
          />
        </motion.div>
      </div>

      {/* Subtle grid overlay */}
      <div
        className="pointer-events-none absolute inset-0 bg-[linear-gradient(rgba(255,255,255,.02)_1px,transparent_1px),linear-gradient(90deg,rgba(255,255,255,.02)_1px,transparent_1px)] bg-size-[64px_64px]"
        aria-hidden
      />
    </section>
  );
}
