"use client";

import { useTranslations, useLocale } from "next-intl";
import { motion } from "motion/react";
import { usePathname } from "@/core/lib/i18n/navigation";
import { routing } from "@/core/lib/i18n/routing";
import { MotionButton } from "@/components/ui/button";

export function Footer() {
  const t = useTranslations("footer");
  const locale = useLocale();
  const pathname = usePathname();

  const switchLocale = (newLocale: string) => {
    const currentPath = pathname === "/" ? "" : pathname;
    const newUrl = `/${newLocale}${currentPath}`;
    window.location.assign(newUrl);
  };

  return (
    <motion.footer
      className="relative border-t border-slate-800 py-8 md:py-10 bg-slate-950 overflow-hidden"
      initial={{ opacity: 0 }}
      whileInView={{ opacity: 1 }}
      viewport={{ once: true }}
      transition={{ duration: 0.6 }}
    >
      {/* Subtle top glow */}
      <motion.div
        className="absolute inset-x-0 top-0 h-px bg-linear-to-r from-transparent via-blue-500/30 to-transparent"
        initial={{ scaleX: 0 }}
        whileInView={{ scaleX: 1 }}
        viewport={{ once: true }}
        transition={{ duration: 1 }}
      />

      {/* Background glow */}
      <div className="absolute inset-0 pointer-events-none">
        <div className="absolute bottom-0 left-1/2 -translate-x-1/2 h-32 w-96 bg-blue-500/5 rounded-full blur-[100px]" />
      </div>

      <div className="container relative z-10 flex flex-col items-center justify-between gap-4 md:flex-row">
        <motion.p
          className="text-center text-sm leading-loose text-slate-400 md:text-left"
          initial={{ opacity: 0, y: 10, filter: "blur(10px)" }}
          whileInView={{ opacity: 1, y: 0, filter: "blur(0px)" }}
          viewport={{ once: true }}
          transition={{ duration: 0.5, delay: 0.2 }}
        >
          © {new Date().getFullYear()} {t("copyright")}
        </motion.p>

        <div className="flex flex-col items-center gap-4 md:flex-row">
          {/* Language Switcher */}
          <motion.div
            className="flex items-center gap-2"
            initial={{ opacity: 0, y: 10, filter: "blur(10px)" }}
            whileInView={{ opacity: 1, y: 0, filter: "blur(0px)" }}
            viewport={{ once: true }}
            transition={{ duration: 0.5, delay: 0.3 }}
          >
            {routing.locales.map((loc) => (
              <MotionButton
                key={loc}
                variant={locale === loc ? "gradient" : "outline"}
                size="sm"
                shine={locale === loc}
                glow={locale === loc}
                onClick={() => switchLocale(loc)}
              >
                {loc.toUpperCase()}
              </MotionButton>
            ))}
          </motion.div>

          {/* Decorative dots */}
          <motion.div
            className="flex items-center gap-2"
            initial={{ opacity: 0 }}
            whileInView={{ opacity: 1 }}
            viewport={{ once: true }}
            transition={{ delay: 0.4 }}
          >
            {[...Array(3)].map((_, i) => (
              <motion.div
                key={i}
                className="h-1.5 w-1.5 rounded-full bg-blue-500/50"
                animate={{
                  opacity: [0.3, 1, 0.3],
                  scale: [0.8, 1.2, 0.8],
                }}
                transition={{
                  duration: 2,
                  repeat: Infinity,
                  delay: i * 0.3,
                  ease: "easeInOut",
                }}
              />
            ))}
          </motion.div>
        </div>
      </div>
    </motion.footer>
  );
}
