"use client";
import { MotionButton } from "@/components/ui/button";
import { BackgroundBeams } from "@/components/ui/background-beams";
import { TextFlip } from "@/components/ui/flip-words";
import { SparklesCanvas } from "@/components/ui/sparkles";
import { Handshake, Leaf, Mail, ChevronDown, Mouse } from "lucide-react";
import { useTranslations } from "next-intl";
import { motion, type Variants, type Easing, useMotionValue, useMotionTemplate } from "framer-motion";
import { useMemo, useCallback } from "react";
import Image from "next/image";
import { ScrollRotate } from "@/components/ui/scroll-rotate";

// Custom easing as tuple for type safety
const customEase: Easing = [0.22, 1, 0.36, 1];

// Animation variants
const containerVariants: Variants = {
  hidden: { opacity: 0 },
  visible: {
    opacity: 1,
    transition: {
      staggerChildren: 0.15,
      delayChildren: 0.3,
    },
  },
};

const fadeUpVariants: Variants = {
  hidden: {
    opacity: 0,
    y: 40,
    filter: "blur(20px)"
  },
  visible: {
    opacity: 1,
    y: 0,
    filter: "blur(0px)",
    transition: {
      duration: 0.8,
      ease: customEase,
    },
  },
};

// Text blur animation variant
const textBlurVariants: Variants = {
  hidden: {
    opacity: 0,
    filter: "blur(20px)",
    y: 20,
  },
  visible: {
    opacity: 1,
    filter: "blur(0px)",
    y: 0,
    transition: {
      duration: 1,
      ease: customEase,
    },
  },
};

const scaleUpVariants: Variants = {
  hidden: {
    opacity: 0,
    scale: 0.8,
  },
  visible: {
    opacity: 1,
    scale: 1,
    transition: {
      duration: 0.5,
      ease: "easeOut" as Easing,
    },
  },
};

const cardVariants: Variants = {
  hidden: {
    opacity: 0,
    y: 50,
    rotateX: -15,
    filter: "blur(15px)",
  },
  visible: {
    opacity: 1,
    y: 0,
    rotateX: 0,
    filter: "blur(0px)",
    transition: {
      duration: 0.8,
      ease: customEase,
    },
  },
};

const floatingVariants: Variants = {
  animate: {
    y: [0, -10, 0],
    transition: {
      duration: 3,
      repeat: Infinity,
      ease: "easeInOut" as Easing,
    },
  },
};

const glowVariants: Variants = {
  animate: {
    boxShadow: [
      "0 0 20px rgba(59, 130, 246, 0.3)",
      "0 0 40px rgba(59, 130, 246, 0.5)",
      "0 0 20px rgba(59, 130, 246, 0.3)",
    ],
    transition: {
      duration: 2,
      repeat: Infinity,
      ease: "easeInOut" as Easing,
    },
  },
};

// Pre-generated particle positions (deterministic)
const particleData = [
  { left: 15, top: 20, duration: 3.5, delay: 0.2 },
  { left: 85, top: 15, duration: 4.2, delay: 0.8 },
  { left: 45, top: 80, duration: 3.8, delay: 1.2 },
  { left: 70, top: 45, duration: 4.5, delay: 0.5 },
  { left: 25, top: 60, duration: 3.2, delay: 1.5 },
  { left: 90, top: 70, duration: 4.0, delay: 0.3 },
  { left: 10, top: 85, duration: 3.6, delay: 1.0 },
  { left: 55, top: 25, duration: 4.3, delay: 0.7 },
  { left: 35, top: 90, duration: 3.4, delay: 1.8 },
  { left: 75, top: 55, duration: 4.1, delay: 0.4 },
  { left: 5, top: 40, duration: 3.9, delay: 1.3 },
  { left: 60, top: 10, duration: 4.4, delay: 0.6 },
  { left: 40, top: 65, duration: 3.3, delay: 1.6 },
  { left: 80, top: 30, duration: 4.2, delay: 0.9 },
  { left: 20, top: 75, duration: 3.7, delay: 1.1 },
  { left: 95, top: 50, duration: 4.0, delay: 0.1 },
  { left: 30, top: 35, duration: 3.5, delay: 1.4 },
  { left: 65, top: 85, duration: 4.3, delay: 0.2 },
  { left: 50, top: 5, duration: 3.8, delay: 1.7 },
  { left: 12, top: 55, duration: 4.1, delay: 0.8 },
];

function PrincipleCard({
  title,
  description,
  icon: Icon,
}: {
  title: string;
  description: string;
  icon: React.ComponentType<{ className?: string }>;
}) {
  const mouseX = useMotionValue(0);
  const mouseY = useMotionValue(0);

  const handleMouseMove = useCallback(
    (event: React.MouseEvent<HTMLDivElement>) => {
      const { currentTarget, clientX, clientY } = event;
      const { left, top } = currentTarget.getBoundingClientRect();
      mouseX.set(clientX - left);
      mouseY.set(clientY - top);
    },
    [mouseX, mouseY]
  );

  return (
    <motion.div
      variants={cardVariants}
      whileHover={{
        scale: 1.02,
        y: -5,
        transition: { duration: 0.2 }
      }}
      whileTap={{ scale: 0.98 }}
      onMouseMove={handleMouseMove}
      className="group/spotlight group relative h-full rounded-xl overflow-hidden border border-slate-700 bg-slate-900/50 backdrop-blur-sm hover:border-blue-500/50 transition-colors"
      style={{ perspective: "1000px" }}
    >
      {/* Spotlight Effect */}
      <motion.div
        className="pointer-events-none absolute -inset-px z-0 rounded-[inherit] opacity-0 transition-opacity duration-300 group-hover/spotlight:opacity-100"
        style={{
          background: useMotionTemplate`
            radial-gradient(
              300px circle at ${mouseX}px ${mouseY}px,
              rgba(59, 130, 246, 0.15),
              transparent 80%
            )
          `,
        }}
      />
      {/* Inner glow */}
      <motion.div
        className="pointer-events-none absolute inset-0 z-0 rounded-[inherit] opacity-0 transition-opacity duration-300 group-hover/spotlight:opacity-100"
        style={{
          background: useMotionTemplate`
            radial-gradient(
              180px circle at ${mouseX}px ${mouseY}px,
              rgba(59, 130, 246, 0.1),
              transparent 60%
            )
          `,
        }}
      />
      <div className="relative z-10 p-6 h-full flex flex-col">
        <motion.div
          className="mb-4"
        >
          <Icon className="w-10 h-10 text-secondary" />
        </motion.div>
        <h3 className="text-xl font-semibold text-white mb-3">{title}</h3>
        <p className="text text-muted-foreground">{description}</p>
      </div>
    </motion.div>
  );
}

function FloatingParticles() {
  const particles = useMemo(() => particleData, []);

  return (
    <motion.div
      className="absolute inset-0 z-10 pointer-events-none"
      initial={{ opacity: 0 }}
      animate={{ opacity: 1 }}
      transition={{ delay: 1, duration: 1 }}
    >
      {particles.map((particle, i) => (
        <motion.div
          key={i}
          className="absolute w-1 h-1 bg-blue-500/30 rounded-full"
          style={{
            left: `${particle.left}%`,
            top: `${particle.top}%`,
          }}
          animate={{
            y: [0, -30, 0],
            opacity: [0.3, 0.8, 0.3],
            scale: [1, 1.5, 1],
          }}
          transition={{
            duration: particle.duration,
            repeat: Infinity,
            delay: particle.delay,
            ease: "easeInOut",
          }}
        />
      ))}
    </motion.div>
  );
}

export function Hero() {
  const t = useTranslations("hero");

  return (
    <section id="hero" className="relative w-full overflow-hidden bg-slate-950 min-h-screen flex flex-col items-center justify-center rounded-lg">
      <motion.div
        className="container relative z-20 py-12 md:py-24 flex flex-col items-center text-center"
        variants={containerVariants}
        initial="hidden"
        animate="visible"
      >
        {/* <motion.div
          variants={scaleUpVariants}
          className="relative"
        >

        </motion.div> */}




        {/* Header Section */}
        <div className="relative z-20 min-h-screen flex flex-col items-center justify-center px-4 text-center">
          <motion.div
            initial={{ opacity: 0, y: 40 }}
            whileInView={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.8 }}
            className=""
          >
            <motion.span
              initial={{ opacity: 0, scale: 0.8 }}
              whileInView={{ opacity: 1, scale: 1 }}
              transition={{ duration: 0.5 }}
              className="inline-block px-4 py-2 mb-8 text-sm font-semibold text-blue-400 bg-blue-500/10 border border-blue-500/20 rounded-full"
            >
              {t("badge")}
            </motion.span>

            {/* Animated Text Flip */}
            <div className="min-w-6xl">
              <TextFlip
                phrases={[
                  { prefix: t("flip1.prefix"), highlight: t("flip1.highlight") },
                  { prefix: t("flip2.prefix"), highlight: t("flip2.highlight") },
                ]}
                duration={4000}
              />
            </div>

            {/* Sparkles Effect */}
            <SparklesCanvas
              className="w-full max-w-120 mx-auto mb-8"
              particleColor="var(--secondary)"
              particleDensity={60}
              minSize={0.5}
              maxSize={1.5}
              speed={0.8}
            />


            <p className="text-xl md:text-2xl text-slate-400 max-w-2xl mx-auto mb-12">
              {t("exploreText")}
            </p>

            {/* Scroll indicator */}
            <motion.div
              animate={{ y: [0, 10, 0] }}
              transition={{ duration: 2, repeat: Infinity }}
              className="flex flex-col items-center gap-2 text-slate-500"
            >
              <span className="text-sm font-medium">{t("scrollToExplore")}</span>
              <svg
                className="w-6 h-6"
                fill="none"
                viewBox="0 0 24 24"
                stroke="currentColor"
              >
                <path
                  strokeLinecap="round"
                  strokeLinejoin="round"
                  strokeWidth={2}
                  d="M19 14l-7 7m0 0l-7-7m7 7V3"
                />
              </svg>
            </motion.div>
          </motion.div>

          {/* Background decoration */}
          <div className="absolute inset-0 overflow-hidden pointer-events-none">
            <div className="absolute top-1/4 left-1/4 w-96 h-96 bg-blue-500/5 rounded-full blur-3xl" />
            <div className="absolute bottom-1/4 right-1/4 w-96 h-96 bg-purple-500/5 rounded-full blur-3xl" />
          </div>
        </div>



        {/* Subtitles with blur effect */}
        <motion.div
          variants={fadeUpVariants}
          className="mt-4 flex flex-col gap-3 max-w-[850px] text-xl text-muted-foreground sm:text-2xl md:text-xl"
        >
          <motion.p

            initial={{ opacity: 0, x: -30, filter: "blur(15px)" }}
            animate={{ opacity: 1, x: 0, filter: "blur(0px)" }}
            transition={{ delay: 0.8, duration: 0.8, ease: customEase }}
          >
            {t("subtitle")}
          </motion.p>
          <motion.p
            initial={{ opacity: 0, x: 30, filter: "blur(15px)" }}
            animate={{ opacity: 1, x: 0, filter: "blur(0px)" }}
            transition={{ delay: 1.1, duration: 0.8, ease: customEase }}
          >
            {t("subtitle2")}
          </motion.p>
        </motion.div>

        {/* CTA Button with animations and blur effect */}
        <motion.div
          initial={{ opacity: 0, y: 30, filter: "blur(15px)" }}
          animate={{ opacity: 1, y: 0, filter: "blur(0px)" }}
          transition={{ delay: 1.4, duration: 0.8, ease: customEase }}
          className="flex gap-4 mt-8"
        >
          <MotionButton
            variant="gradient-outline"
            size="xl"
            iconLeft={<Mail className="w-6 h-6" />}
            className="px-8 py-6"
            whileHover={{ scale: 1.05 }}
            whileTap={{ scale: 0.95 }}
            animate={{
              boxShadow: [
                "0 0 20px rgba(6, 182, 212, 0.3)",
                "0 0 40px rgba(6, 182, 212, 0.5)",
                "0 0 20px rgba(6, 182, 212, 0.3)",
              ],
            }}
            transition={{
              boxShadow: {
                duration: 2,
                repeat: Infinity,
                ease: "easeInOut",
              },
            }}
          >
            {t("cta")}
          </MotionButton>
        </motion.div>

        {/* Principle Cards with stagger animation */}
        <motion.div
          className="mt-8 grid grid-cols-1 md:grid-cols-2 gap-8 max-w-4xl w-full text-left"
          variants={containerVariants}
        >
          <PrincipleCard
            title={t("principles.collaboration.title")}
            description={t("principles.collaboration.description")}
            icon={Handshake}
          />
          <PrincipleCard
            title={t("principles.sustainability.title")}
            description={t("principles.sustainability.description")}
            icon={Leaf}
          />
        </motion.div>

        {/* Scroll indicator */}
        <motion.div
          className="absolute bottom-8 left-1/2 -translate-x-1/2"
          variants={floatingVariants}
          animate="animate"
          initial={{ opacity: 0, filter: "blur(10px)" }}
          whileInView={{ opacity: 1, filter: "blur(0px)" }}
          transition={{ delay: 2, duration: 0.6 }}
        >
          <motion.div
            className="flex flex-col items-center gap-2 text-muted-foreground cursor-pointer"
            whileHover={{ scale: 1.1 }}
            onClick={() => {
              document.getElementById("services")?.scrollIntoView({ behavior: "smooth" });
            }}
          >
            <span className="text-base">
              <Mouse />
            </span>
            <ChevronDown className="w-6 h-6" />
          </motion.div>
        </motion.div>
      </motion.div>

      {/* Animated background particles */}
      <FloatingParticles />


      <BackgroundBeams className="absolute top-0 left-0 w-full h-full z-0" />
    </section>
  );
}
