"use client";

import { useMemo, useCallback } from "react";
import { useTranslations } from "next-intl";
import { motion, useMotionValue, useMotionTemplate } from "motion/react";
import { cn } from "@/core/lib/utils";
import {
  HeartPulse,
  Landmark,
  TrendingUp,
  Plane,
  ShoppingBag,
  Factory,
  Truck,
  Monitor,
  type LucideIcon,
} from "lucide-react";

// --- Types ---

type IndustryItem = {
  name: string;
  description: string;
};

function isIndustryItems(value: unknown): value is IndustryItem[] {
  if (!Array.isArray(value)) return false;
  return value.every((item) => {
    if (!item || typeof item !== "object") return false;
    const asAny = item as Record<string, unknown>;
    return (
      typeof asAny.name === "string" && typeof asAny.description === "string"
    );
  });
}

// Industry icons mapping
const industryIcons: LucideIcon[] = [
  HeartPulse, // Health Care
  Landmark, // Banking
  TrendingUp, // Financial
  Plane, // Tourism
  ShoppingBag, // Retail
  Factory, // Manufacturing
  Truck, // Logistics & Transportation
  Monitor, // Information Technologies
];

// Gradient colors for each industry
const industryGradients = [
  "from-rose-500/20 via-pink-500/10 to-transparent", // Health Care
  "from-blue-500/20 via-indigo-500/10 to-transparent", // Banking
  "from-emerald-500/20 via-green-500/10 to-transparent", // Financial
  "from-amber-500/20 via-orange-500/10 to-transparent", // Tourism
  "from-purple-500/20 via-violet-500/10 to-transparent", // Retail
  "from-slate-500/20 via-zinc-500/10 to-transparent", // Manufacturing
  "from-cyan-500/20 via-teal-500/10 to-transparent", // Logistics
  "from-blue-600/20 via-sky-500/10 to-transparent", // IT
];

const iconColors = [
  "text-rose-400 group-hover:text-rose-300",
  "text-blue-400 group-hover:text-blue-300",
  "text-emerald-400 group-hover:text-emerald-300",
  "text-amber-400 group-hover:text-amber-300",
  "text-purple-400 group-hover:text-purple-300",
  "text-slate-400 group-hover:text-slate-300",
  "text-cyan-400 group-hover:text-cyan-300",
  "text-sky-400 group-hover:text-sky-300",
];

const borderColors = [
  "border-rose-500/20 hover:border-rose-500/40",
  "border-blue-500/20 hover:border-blue-500/40",
  "border-emerald-500/20 hover:border-emerald-500/40",
  "border-amber-500/20 hover:border-amber-500/40",
  "border-purple-500/20 hover:border-purple-500/40",
  "border-slate-500/20 hover:border-slate-500/40",
  "border-cyan-500/20 hover:border-cyan-500/40",
  "border-sky-500/20 hover:border-sky-500/40",
];

// --- Components ---

function IndustryCard({
  industry,
  index,
  icon: Icon,
}: {
  industry: IndustryItem;
  index: number;
  icon: LucideIcon;
}) {
  const mouseX = useMotionValue(0);
  const mouseY = useMotionValue(0);

  const handleMouseMove = useCallback(
    (event: React.MouseEvent<HTMLDivElement>) => {
      const { currentTarget, clientX, clientY } = event;
      const { left, top } = currentTarget.getBoundingClientRect();
      mouseX.set(clientX - left);
      mouseY.set(clientY - top);
    },
    [mouseX, mouseY]
  );

  return (
    <motion.div
      className={cn(
        "group/spotlight group relative overflow-hidden rounded-2xl border bg-slate-950/50 backdrop-blur-sm p-6 transition-all duration-500",
        borderColors[index]
      )}
      onMouseMove={handleMouseMove}
      initial={{ opacity: 0, y: 40, filter: "blur(15px)" }}
      whileInView={{ opacity: 1, y: 0, filter: "blur(0px)" }}
      viewport={{ once: true, amount: 0.2 }}
      transition={{ delay: index * 0.1, duration: 0.6, ease: [0.22, 1, 0.36, 1] }}
      whileHover={{ y: -8, scale: 1.02, transition: { duration: 0.3 } }}
    >
      {/* Spotlight Effect */}
      <motion.div
        className="pointer-events-none absolute -inset-px z-0 rounded-[inherit] opacity-0 transition-opacity duration-300 group-hover/spotlight:opacity-100"
        style={{
          background: useMotionTemplate`
            radial-gradient(
              300px circle at ${mouseX}px ${mouseY}px,
              rgba(59, 130, 246, 0.12),
              transparent 80%
            )
          `,
        }}
      />

      {/* Background gradient */}
      <div
        className={cn(
          "absolute inset-0 bg-linear-to-br opacity-0 transition-opacity duration-500 group-hover:opacity-100",
          industryGradients[index]
        )}
      />

      {/* Floating particles effect */}
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        <motion.div
          className="absolute h-32 w-32 rounded-full bg-white/5 blur-2xl"
          animate={{
            x: [0, 30, 0],
            y: [0, -20, 0],
          }}
          transition={{
            duration: 8,
            repeat: Infinity,
            ease: "easeInOut",
          }}
          style={{ top: "20%", right: "-10%" }}
        />
      </div>

      <div className="relative z-10">
        {/* Icon container */}
        <motion.div
          className={cn(
            "mb-4 flex h-14 w-14 items-center justify-center rounded-xl bg-slate-800/50 backdrop-blur-sm transition-all duration-300 group-hover:scale-110 group-hover:bg-slate-800/80",
            iconColors[index]
          )}
          whileHover={{ rotate: [0, -5, 5, 0] }}
          transition={{ duration: 0.4 }}
        >
          <Icon className="h-7 w-7" />
        </motion.div>

        {/* Content */}
        <h3 className="mb-2 text-lg font-semibold text-white transition-transform duration-300 group-hover:translate-x-1">
          {industry.name}
        </h3>
        <p className="text-sm leading-relaxed text-slate-400 transition-colors duration-300 group-hover:text-slate-300">
          {industry.description}
        </p>

        {/* Decorative line */}
        <motion.div
          className={cn(
            "mt-4 h-0.5 w-0 rounded-full transition-all duration-500 group-hover:w-full",
            index % 2 === 0
              ? "bg-linear-to-r from-blue-500 to-purple-500"
              : "bg-linear-to-r from-emerald-500 to-cyan-500"
          )}
        />
      </div>
    </motion.div>
  );
}

// --- Main Component ---

export function IndustryExpertise() {
  const t = useTranslations("industryExpertise");

  const industries = useMemo(() => {
    const raw = t.raw("industries");
    return isIndustryItems(raw) ? raw : [];
  }, [t]);

  return (
    <section id="industry-expertise" className="relative bg-slate-950 py-24 md:py-32 overflow-hidden">
      {/* Background decorations */}
      <div className="absolute inset-0 pointer-events-none">
        {/* Grid pattern */}
        <div
          className="absolute inset-0 opacity-[0.02]"
          style={{
            backgroundImage: `linear-gradient(rgba(255,255,255,0.1) 1px, transparent 1px),
              linear-gradient(90deg, rgba(255,255,255,0.1) 1px, transparent 1px)`,
            backgroundSize: "50px 50px",
          }}
        />

        {/* Ambient glows */}
        <div className="absolute top-0 left-1/4 h-125 w-125 bg-blue-600/10 rounded-full blur-[150px]" />
        <div className="absolute bottom-0 right-1/4 h-100 w-100 bg-purple-600/10 rounded-full blur-[120px]" />
      </div>

      {/* Top border glow */}
      <div className="absolute inset-x-0 top-0 h-px bg-linear-to-r from-transparent via-slate-500/30 to-transparent" />

      <div className="container relative z-10">
        {/* Header */}
        <motion.div
          className="mx-auto max-w-3xl text-center mb-16"
          initial={{ opacity: 0, y: 30, filter: "blur(15px)" }}
          whileInView={{ opacity: 1, y: 0, filter: "blur(0px)" }}
          viewport={{ once: true }}
          transition={{ duration: 0.8, ease: [0.22, 1, 0.36, 1] }}
        >
          <motion.h2
            className="text-3xl font-bold tracking-tight text-white md:text-4xl lg:text-5xl"
            initial={{ opacity: 0, y: 20, filter: "blur(12px)" }}
            whileInView={{ opacity: 1, y: 0, filter: "blur(0px)" }}
            viewport={{ once: true }}
            transition={{ duration: 0.6, delay: 0.1 }}
          >
            {t("title")}
          </motion.h2>
          <motion.p
            className="mt-4 text-lg text-slate-400 md:text-xl"
            initial={{ opacity: 0, y: 15, filter: "blur(10px)" }}
            whileInView={{ opacity: 1, y: 0, filter: "blur(0px)" }}
            viewport={{ once: true }}
            transition={{ duration: 0.6, delay: 0.2 }}
          >
            {t("subtitle")}
          </motion.p>
        </motion.div>

        {/* Industry Grid */}
        <div className="grid gap-4 sm:grid-cols-2 lg:grid-cols-4">
          {industries.map((industry, index) => (
            <IndustryCard
              key={index}
              industry={industry}
              index={index}
              icon={industryIcons[index] ?? Monitor}
            />
          ))}
        </div>

        {/* Bottom decorative element */}
        <motion.div
          className="mt-16 flex justify-center"
          initial={{ opacity: 0 }}
          whileInView={{ opacity: 1 }}
          viewport={{ once: true }}
          transition={{ delay: 0.5, duration: 0.6 }}
        >
          <div className="flex items-center gap-3">
            <div className="h-px w-12 bg-linear-to-r from-transparent to-slate-600" />
            <div className="flex gap-1">
              {[...Array(3)].map((_, i) => (
                <motion.div
                  key={i}
                  className="h-1.5 w-1.5 rounded-full bg-blue-500"
                  animate={{ opacity: [0.3, 1, 0.3] }}
                  transition={{
                    duration: 1.5,
                    repeat: Infinity,
                    delay: i * 0.2,
                  }}
                />
              ))}
            </div>
            <div className="h-px w-12 bg-linear-to-l from-transparent to-slate-600" />
          </div>
        </motion.div>
      </div>
    </section>
  );
}
