"use client";

import { useTranslations } from "next-intl";
import { motion } from "motion/react";
import Image from "next/image";
import { cn } from "@/core/lib/utils";
import type { StrapiPartner } from "@/core/lib/api";
import { getStrapiMediaUrl } from "@/core/lib/api";

export interface PartnerDisplay {
  name: string;
  logo: string | null;
}

const fallbackPartners: PartnerDisplay[] = [
  { name: "Microsoft", logo: "/images/partners/microsoft.svg" },
  { name: "AWS", logo: "/images/partners/aws.svg" },
  { name: "Google Cloud", logo: "/images/partners/google-cloud.svg" },
  { name: "Oracle", logo: "/images/partners/oracle.svg" },
  { name: "SAP", logo: "/images/partners/sap.svg" },
  { name: "Salesforce", logo: "/images/partners/salesforce.svg" },
  { name: "IBM", logo: "/images/partners/ibm.svg" },
  { name: "Cisco", logo: "/images/partners/cisco.svg" },
];

function toDisplayPartners(partners: StrapiPartner[]): PartnerDisplay[] {
  return partners.map((p) => ({
    name: p.name ?? "Partner",
    logo: p.logo?.url ? getStrapiMediaUrl(p.logo.url) : null,
  }));
}

// --- Components ---

function LogoMarquee({
  logos,
  direction = "left",
  speed = 25,
}: {
  logos: PartnerDisplay[];
  direction?: "left" | "right";
  speed?: number;
}) {
  return (
    <div className="relative flex overflow-hidden">
      {/* Gradient masks */}
      <div className="pointer-events-none absolute left-0 top-0 z-10 h-full w-32 bg-linear-to-r from-slate-950 to-transparent" />
      <div className="pointer-events-none absolute right-0 top-0 z-10 h-full w-32 bg-linear-to-l from-slate-950 to-transparent" />

      <motion.div
        className="flex shrink-0 gap-12"
        animate={{
          x: direction === "left" ? ["0%", "-50%"] : ["-50%", "0%"],
        }}
        transition={{
          x: {
            duration: speed,
            repeat: Infinity,
            ease: "linear",
          },
        }}
      >
        {/* Double the logos for seamless loop */}
        {[...logos, ...logos].map((partner, index) => (
          <div
            key={`${partner.name}-${index}`}
            className="group relative flex h-20 w-40 shrink-0 items-center justify-center rounded-xl border border-slate-800/50 bg-slate-900/50 px-6 transition-all duration-300 hover:border-slate-700 hover:bg-slate-800/50"
          >
            <div className="flex h-full w-full items-center justify-center">
              {partner.logo ? (
                <Image
                  src={partner.logo}
                  alt={partner.name}
                  width={120}
                  height={40}
                  className="object-contain max-h-10 opacity-60 group-hover:opacity-100 transition-opacity"
                />
              ) : (
                <span className="text-lg font-semibold text-slate-400 transition-colors duration-300 group-hover:text-slate-200">
                  {partner.name}
                </span>
              )}
            </div>

            {/* Glow effect on hover */}
            <div className="pointer-events-none absolute -inset-px rounded-xl opacity-0 transition-opacity duration-300 group-hover:opacity-100">
              <div className="absolute inset-0 rounded-xl bg-linear-to-r from-blue-500/10 via-transparent to-purple-500/10" />
            </div>
          </div>
        ))}
      </motion.div>
    </div>
  );
}

function PartnerGrid({ partners }: { partners: PartnerDisplay[] }) {
  return (
    <div className="grid grid-cols-2 gap-4 sm:grid-cols-3 md:grid-cols-4 lg:hidden">
      {partners.map((partner, index) => (
        <motion.div
          key={`${partner.name}-${index}`}
          className="group flex h-20 items-center justify-center rounded-xl border border-slate-800/50 bg-slate-900/50 px-4 transition-all duration-300 hover:border-slate-700 hover:bg-slate-800/50"
          initial={{ opacity: 0, y: 25, filter: "blur(12px)" }}
          whileInView={{ opacity: 1, y: 0, filter: "blur(0px)" }}
          viewport={{ once: true }}
          transition={{ delay: index * 0.06, duration: 0.5, ease: [0.22, 1, 0.36, 1] }}
          whileHover={{ y: -3, scale: 1.02 }}
        >
          {partner.logo ? (
            <Image
              src={partner.logo}
              alt={partner.name}
              width={100}
              height={36}
              className="object-contain max-h-9 opacity-60 group-hover:opacity-100 transition-opacity"
            />
          ) : (
            <span className="text-sm font-semibold text-slate-400 transition-colors duration-300 group-hover:text-slate-200 text-center">
              {partner.name}
            </span>
          )}
        </motion.div>
      ))}
    </div>
  );
}

// --- Main Component ---

export interface PartnersProps {
  partners: StrapiPartner[];
}

export function Partners({ partners: apiPartners = [] }: PartnersProps) {
  const t = useTranslations("partners");
  const partners = apiPartners.length > 0 ? toDisplayPartners(apiPartners) : fallbackPartners;

  return (
    <section id="partners" className="relative bg-slate-950 py-24 md:py-32 overflow-hidden">
      {/* Background elements */}
      <div className="absolute inset-0 pointer-events-none">
        <div
          className="absolute inset-0 opacity-[0.015]"
          style={{
            backgroundImage: `radial-gradient(circle at 1px 1px, rgba(255,255,255,0.3) 1px, transparent 0)`,
            backgroundSize: "40px 40px",
          }}
        />
      </div>

      {/* Top border */}
      <div className="absolute inset-x-0 top-0 h-px bg-linear-to-r from-transparent via-slate-600/30 to-transparent" />

      <div className="container relative z-10">
        {/* Header */}
        <motion.div
          className="mx-auto max-w-3xl text-center mb-16"
          initial={{ opacity: 0, y: 30, filter: "blur(15px)" }}
          whileInView={{ opacity: 1, y: 0, filter: "blur(0px)" }}
          viewport={{ once: true }}
          transition={{ duration: 0.8, ease: [0.22, 1, 0.36, 1] }}
        >
          <motion.h2
            className="text-3xl font-bold tracking-tight text-white md:text-4xl lg:text-5xl"
            initial={{ opacity: 0, y: 20, filter: "blur(12px)" }}
            whileInView={{ opacity: 1, y: 0, filter: "blur(0px)" }}
            viewport={{ once: true }}
            transition={{ duration: 0.6, delay: 0.1 }}
          >
            {t("title")}
          </motion.h2>
          <motion.p
            className="mt-4 text-lg text-slate-400 md:text-xl"
            initial={{ opacity: 0, y: 15, filter: "blur(10px)" }}
            whileInView={{ opacity: 1, y: 0, filter: "blur(0px)" }}
            viewport={{ once: true }}
            transition={{ duration: 0.6, delay: 0.2 }}
          >
            {t("subtitle")}
          </motion.p>
        </motion.div>

        {/* Desktop: Infinite scroll marquee */}
        <div className="hidden lg:block space-y-6">
          <LogoMarquee logos={partners} direction="left" speed={30} />
          <LogoMarquee logos={partners.slice().reverse()} direction="right" speed={35} />
        </div>

        {/* Mobile/Tablet: Grid layout */}
        <PartnerGrid partners={partners} />

        {/* Stats */}
        <motion.div
          className="mt-16 grid grid-cols-2 gap-6 md:grid-cols-4"
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ delay: 0.3, duration: 0.6 }}
        >
          {[
            { value: `${partners.length}+`, label: "Technology Partners" },
            { value: "50+", label: "Certifications" },
            { value: "99%", label: "Client Satisfaction" },
            { value: "24/7", label: "Support Available" },
          ].map((stat, index) => (
            <motion.div
              key={stat.label}
              className="group relative overflow-hidden rounded-xl border border-slate-800/50 bg-slate-900/30 p-6 text-center transition-all duration-300 hover:border-slate-700"
              whileHover={{ y: -2 }}
            >
              <div className="text-3xl font-bold text-white md:text-4xl">
                {stat.value}
              </div>
              <div className="mt-1 text-sm text-slate-400">{stat.label}</div>

              {/* Decorative gradient */}
              <div className="absolute inset-0 bg-linear-to-br from-blue-500/5 via-transparent to-purple-500/5 opacity-0 transition-opacity duration-300 group-hover:opacity-100" />
            </motion.div>
          ))}
        </motion.div>
      </div>

      {/* Bottom decorative element */}
      <motion.div
        className="absolute bottom-8 left-1/2 -translate-x-1/2"
        initial={{ opacity: 0 }}
        whileInView={{ opacity: 1 }}
        viewport={{ once: true }}
        transition={{ delay: 0.5 }}
      >
        <div className="flex items-center gap-2">
          <div className="h-px w-8 bg-linear-to-r from-transparent to-slate-600" />
          <div className="h-1 w-1 rounded-full bg-slate-600" />
          <div className="h-px w-8 bg-linear-to-l from-transparent to-slate-600" />
        </div>
      </motion.div>
    </section>
  );
}
