"use client";

import { useState } from "react";
import { motion, AnimatePresence, type Variants, type Easing } from "motion/react";
import { cn } from "@/core/lib/utils";
import { ProjectCard, Project } from "./project-card";

// Custom easing
const customEase: Easing = [0.22, 1, 0.36, 1];

const containerVariants: Variants = {
  hidden: { opacity: 0 },
  visible: {
    opacity: 1,
    transition: {
      staggerChildren: 0.15,
      delayChildren: 0.1,
    },
  },
};

const headerVariants: Variants = {
  hidden: {
    opacity: 0,
    y: 40,
    filter: "blur(15px)"
  },
  visible: {
    opacity: 1,
    y: 0,
    filter: "blur(0px)",
    transition: {
      duration: 0.8,
      ease: customEase,
    },
  },
};

const tabContainerVariants: Variants = {
  hidden: {
    opacity: 0,
    y: 20,
    filter: "blur(10px)"
  },
  visible: {
    opacity: 1,
    y: 0,
    filter: "blur(0px)",
    transition: {
      duration: 0.6,
      ease: customEase,
      delay: 0.3,
    },
  },
};

interface PortfolioGridProps {
  title: string;
  subtitle: string;
  projects: Project[];
  services: string[];
}

export function PortfolioGrid({ title, subtitle, projects, services }: PortfolioGridProps) {
  const [activeTab, setActiveTab] = useState("All");
  const [hovered, setHovered] = useState<string | null>(null);

  const filteredProjects = activeTab === "All"
    ? projects
    : projects.filter(project => project.services.includes(activeTab));

  return (
    <section id="portfolio" className="relative py-24 md:py-32 bg-slate-950 overflow-hidden">
      {/* Background Elements */}
      <div
        className="pointer-events-none absolute inset-0 bg-[linear-gradient(rgba(255,255,255,.02)_1px,transparent_1px),linear-gradient(90deg,rgba(255,255,255,.02)_1px,transparent_1px)] bg-size-[64px_64px]"
        aria-hidden
      />

      <motion.div
        className="absolute inset-x-0 top-0 h-px bg-gradient-to-r from-transparent via-primary/50 to-transparent z-10"
        initial={{ scaleX: 0, opacity: 0 }}
        whileInView={{ scaleX: 1, opacity: 1 }}
        viewport={{ once: true }}
        transition={{ duration: 1.2, ease: "easeOut" }}
      />

      <div className="absolute inset-0 overflow-hidden pointer-events-none z-0">
        <motion.div
          className="absolute top-[20%] right-[10%] w-[600px] h-[600px] bg-indigo-600/10 rounded-full blur-[120px]"
          animate={{
            scale: [1, 1.2, 1],
            opacity: [0.3, 0.5, 0.3],
          }}
          transition={{
            duration: 10,
            repeat: Infinity,
            ease: "easeInOut",
          }}
        />
        <motion.div
          className="absolute bottom-[10%] left-[5%] w-[500px] h-[500px] bg-blue-600/10 rounded-full blur-[100px]"
          animate={{
            scale: [1.2, 1, 1.2],
            opacity: [0.3, 0.5, 0.3],
          }}
          transition={{
            duration: 10,
            repeat: Infinity,
            ease: "easeInOut",
            delay: 3,
          }}
        />
      </div>

      <div className="container relative z-10">
        <motion.div
          className="mx-auto flex max-w-[980px] flex-col items-center gap-4 text-center mb-16"
          variants={containerVariants}
          initial="hidden"
          whileInView="visible"
          viewport={{ once: true, amount: 0.3 }}
        >
          <motion.h2
            className="text-3xl md:text-5xl font-bold bg-clip-text text-transparent bg-gradient-to-b from-white to-white/70"
            variants={headerVariants}
          >
            {title}
          </motion.h2>
          <motion.p
            className="text-slate-400 text-lg md:text-xl max-w-2xl"
            variants={headerVariants}
          >
            {subtitle}
          </motion.p>
        </motion.div>

        {/* Tabs */}
        <motion.div
          className="flex flex-wrap justify-center gap-2 mb-12"
          variants={tabContainerVariants}
          initial="hidden"
          whileInView="visible"
          viewport={{ once: true }}
        >
          {["All", ...services].map((service, index) => (
            <motion.button
              key={service}
              onClick={() => setActiveTab(service)}
              className={cn(
                "relative px-4 py-2 rounded-full text-sm font-medium transition-colors duration-300",
                activeTab === service
                  ? "text-white"
                  : "text-slate-400 hover:text-white"
              )}
              initial={{ opacity: 0, y: 10 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: 0.4 + index * 0.05 }}
              whileHover={{ scale: 1.05 }}
              whileTap={{ scale: 0.95 }}
            >
              {activeTab === service && (
                <motion.div
                  layoutId="activeTab"
                  className="absolute inset-0 bg-blue-600/20 border border-primary/50 rounded-full backdrop-blur-sm"
                  transition={{ type: "spring", bounce: 0.2, duration: 0.6 }}
                />
              )}
              <span className="relative z-10">{service}</span>
            </motion.button>
          ))}
        </motion.div>

        {/* Grid */}
        <motion.div
          layout
          className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6"
          onMouseLeave={() => setHovered(null)}
        >
          <AnimatePresence mode="popLayout">
            {filteredProjects.map((project, index) => (
              <motion.div
                key={project.id}
                initial={{ opacity: 0, y: 30, filter: "blur(15px)" }}
                animate={{ opacity: 1, y: 0, filter: "blur(0px)" }}
                exit={{ opacity: 0, scale: 0.9, filter: "blur(10px)" }}
                transition={{
                  duration: 0.5,
                  delay: index * 0.08,
                  ease: customEase
                }}
                layout
              >
                <ProjectCard
                  project={project}
                  hovered={hovered}
                  setHovered={setHovered}
                />
              </motion.div>
            ))}
          </AnimatePresence>
        </motion.div>

        {filteredProjects.length === 0 && (
          <motion.div
            className="text-center py-20 text-slate-500"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.4 }}
          >
            No projects found for this category.
          </motion.div>
        )}
      </div>
    </section>
  );
}
