import { getTranslations } from "next-intl/server";
import { PortfolioGrid } from "./portfolio-grid";
import type { Project } from "./project-card";
import type { StrapiProject } from "@/core/lib/api";
import { getStrapiMediaUrl } from "@/core/lib/api";

interface PortfolioProps {
  projects: StrapiProject[];
}

function mapStrapiProjectsToPortfolio(projects: StrapiProject[]): {
  projects: Project[];
  filters: string[];
} {
  const seenFilters = new Set<string>();

  const mapped: Project[] = projects.map((p) => {
    const categories = p.categories ?? [];
    const categoryName =
      p.service?.name ??
      categories[0]?.name ??
      "Project";

    const services: string[] = [];

    if (p.service?.name) {
      services.push(p.service.name);
      seenFilters.add(p.service.name);
    }

    for (const c of categories) {
      if (c.name) {
        services.push(c.name);
        seenFilters.add(c.name);
      }
    }

    if (services.length === 0) {
      services.push("Project");
      seenFilters.add("Project");
    }

    const thumbnail =
      p.cover_image?.url
        ? getStrapiMediaUrl(p.cover_image.url)
        : "/images/projects/placeholder.png";

    return {
      id: p.documentId,
      title: p.title,
      category: categoryName,
      thumbnail,
      services,
      link: p.slug ? `/projects/${p.slug}` : "#",
      slug: p.slug,
    };
  });

  return {
    projects: mapped,
    filters: Array.from(seenFilters),
  };
}

export async function Portfolio({ projects }: PortfolioProps) {
  const t = await getTranslations("portfolio");

  const { projects: mappedProjects, filters } = mapStrapiProjectsToPortfolio(projects);

  return (
    <PortfolioGrid
      title={t("title")}
      subtitle={t("subtitle")}
      projects={mappedProjects}
      services={filters}
    />
  );
}
