"use client";

import { useCallback } from "react";
import { motion, useMotionValue, useMotionTemplate } from "motion/react";
import Image from "next/image";
import Link from "next/link";
import { cn } from "@/core/lib/utils";
import { ArrowUpRight } from "lucide-react";

export interface Project {
    id: string;
    title: string;
    category: string;
    thumbnail: string;
    services: string[];
    link?: string;
    slug?: string;
}

interface ProjectCardProps {
    project: Project;
    className?: string;
    hovered?: string | null;
    setHovered?: (id: string | null) => void;
}

export function ProjectCard({ project, className, hovered, setHovered }: ProjectCardProps) {
    const isHovered = hovered === project.id;
    const isDimmed = hovered !== null && !isHovered;
    const mouseX = useMotionValue(0);
    const mouseY = useMotionValue(0);

    const handleMouseMove = useCallback(
        (event: React.MouseEvent<HTMLDivElement>) => {
            const { currentTarget, clientX, clientY } = event;
            const { left, top } = currentTarget.getBoundingClientRect();
            mouseX.set(clientX - left);
            mouseY.set(clientY - top);
        },
        [mouseX, mouseY]
    );

    return (
        <motion.div
            layout
            initial={{ opacity: 0, scale: 0.9 }}
            animate={{
                opacity: isDimmed ? 0.5 : 1,
                scale: isHovered ? 1.02 : 1,
                filter: isDimmed ? "blur(1px) grayscale(50%)" : "none"
            }}
            exit={{ opacity: 0, scale: 0.9 }}
            transition={{ duration: 0.3 }}
            onMouseEnter={() => setHovered && setHovered(project.id)}
            onMouseLeave={() => setHovered && setHovered(null)}
            onMouseMove={handleMouseMove}
            className={cn(
                "group/spotlight group relative h-full min-h-[400px] w-full cursor-pointer overflow-hidden rounded-3xl border border-slate-800 bg-slate-950/80 transition-all duration-500 hover:border-slate-600/50 hover:shadow-2xl hover:shadow-primary/10",
                className
            )}
        >
            {/* Spotlight Effect */}
            <motion.div
                className="pointer-events-none absolute -inset-px z-25 rounded-[inherit] opacity-0 transition-opacity duration-300 group-hover/spotlight:opacity-100"
                style={{
                    background: useMotionTemplate`
                        radial-gradient(
                            450px circle at ${mouseX}px ${mouseY}px,
                            rgba(59, 130, 246, 0.15),
                            transparent 80%
                        )
                    `,
                }}
            />

            <Link href={project.link || "#"} className="block h-full w-full relative">
                {/* Background Image Layer - Sharp by default */}
                <div className="absolute inset-0 z-0 h-full w-full">
                    <Image
                        src={project.thumbnail}
                        alt={project.title}
                        fill
                        className="object-cover transition-transform duration-700 ease-out group-hover:scale-110"
                    />
                </div>

                {/* Gradient Overlay - Disappears on hover */}
                <div className="absolute inset-0 z-10 bg-gradient-to-t from-black via-black/50 to-transparent opacity-90 transition-opacity duration-500  hover:opacity-75" />

                {/* Top Content: Category Chip */}
                <div className="absolute top-0 left-0 right-0 z-20 p-6 flex justify-between items-start">
                    <span className="inline-block rounded-full bg-slate-900/50 backdrop-blur-md px-3 py-1 text-xs font-semibold uppercase tracking-wider text-white border border-white/10 group-hover:bg-primary group-hover:border-primary transition-colors duration-300">
                        {project.category}
                    </span>

                    {/* Arrow Icon */}
                    <div className="flex h-10 w-10 items-center justify-center rounded-full bg-slate-900/50 backdrop-blur-md border border-white/10 text-white opacity-0 -translate-y-2 transition-all duration-300 group-hover:opacity-100 group-hover:translate-y-0 group-hover:bg-white group-hover:text-black">
                        <ArrowUpRight className="h-5 w-5" />
                    </div>
                </div>

                {/* Bottom Content */}
                <div className="absolute bottom-0 left-0 right-0 z-20 p-6 sm:p-8 flex flex-col justify-end">
                    <div className="transform transition-transform duration-500 group-hover:translate-y-2">
                        <h3 className="text-2xl font-bold tracking-tight text-white mb-2 group-hover:text-transparent group-hover:bg-clip-text group-hover:bg-gradient-to-r group-hover:from-white group-hover:to-slate-300">
                            {project.title}
                        </h3>

                        {/* Services Tags (Optional, keeping minimal as per request) */}
                        <div className="flex flex-wrap gap-2 opacity-70 group-hover:opacity-100 transition-opacity duration-300">
                            {project.services.slice(0, 3).map((s, i) => (
                                <span key={i} className="text-sm text-slate-300">
                                    {s} {i < Math.min(project.services.length, 3) - 1 && "•"}
                                </span>
                            ))}
                        </div>
                    </div>
                </div>
            </Link>
        </motion.div>
    );
}
