"use client";

import { useMemo, useState, useEffect, useRef, useCallback } from "react";
import { useTranslations } from "next-intl";
import { motion, AnimatePresence, useScroll, useTransform, useMotionValue, useMotionTemplate } from "motion/react";
import Image from "next/image";
import { cn } from "@/core/lib/utils";
import type { StrapiService } from "@/core/lib/api";
import { getStrapiMediaUrl } from "@/core/lib/api";
import {
  Code2,
  AppWindow,
  Palette,
  BriefcaseBusiness,
  BarChart3,
  Headset,
  Server,
  Brain,
  LifeBuoy,
  Wifi,
  ShieldCheck,
  Check,
  X,
  ArrowRight,
  type LucideIcon,
} from "lucide-react";
import { ScrollRotate } from "@/components/ui/scroll-rotate";

// --- Types ---

export type ServiceItem = {
  title: string;
  description: string;
  offerings: string[];
  imageUrl?: string | null;
};

function isServiceItems(value: unknown): value is ServiceItem[] {
  if (!Array.isArray(value)) return false;
  return value.every((item) => {
    if (!item || typeof item !== "object") return false;
    const asAny = item as Record<string, unknown>;
    return (
      typeof asAny.title === "string" &&
      typeof asAny.description === "string" &&
      Array.isArray(asAny.offerings) &&
      (asAny.offerings as unknown[]).every((o) => typeof o === "string")
    );
  });
}

function strapiServiceToItem(s: StrapiService, index: number): ServiceItem {
  const desc = s.short_description ?? (typeof s.full_description === "string" ? s.full_description.replace(/<[^>]*>/g, "").slice(0, 300) : "") ?? "";
  const offerings = (s.features ?? [])
    .map((f) => f.title ?? f.description)
    .filter((x): x is string => typeof x === "string" && x.length > 0);
  if (offerings.length === 0 && desc) offerings.push(desc.slice(0, 80) + (desc.length > 80 ? "…" : ""));
  return {
    title: s.name,
    description: desc,
    offerings: offerings.length ? offerings : ["Custom solutions", "Expert support"],
    imageUrl: s.image?.size ? getStrapiMediaUrl(s.image.url) : null,
  };
}



const serviceIcons: LucideIcon[] = [
  Code2,
  AppWindow,
  Palette,
  BriefcaseBusiness,
  BarChart3,
  Headset,
  Server,
  Brain,
  LifeBuoy,
  Wifi,
  ShieldCheck,
];

// --- Components ---

function ServiceCard({
  service,
  index,
  icon: Icon,
  onClick,
  className,
}: {
  service: ServiceItem;
  index: number;
  icon: LucideIcon;
  onClick: () => void;
  className?: string;
}) {
  const mouseX = useMotionValue(0);
  const mouseY = useMotionValue(0);

  const handleMouseMove = useCallback(
    (event: React.MouseEvent<HTMLDivElement>) => {
      const { currentTarget, clientX, clientY } = event;
      const { left, top } = currentTarget.getBoundingClientRect();
      mouseX.set(clientX - left);
      mouseY.set(clientY - top);
    },
    [mouseX, mouseY]
  );

  return (
    <motion.div
      onClick={onClick}
      onMouseMove={handleMouseMove}
      className={cn(
        "group/spotlight group relative h-full flex-shrink-0 cursor-pointer overflow-hidden rounded-3xl border border-slate-800 bg-slate-950/80 transition-all duration-500 hover:border-slate-600/50",
        className
      )}
      initial="initial"
      whileInView="visible"
      whileHover="hover"
      viewport={{ once: true, amount: 0.1 }}
      variants={{
        initial: { opacity: 0, y: 20 },
        visible: {
          opacity: 1,
          y: 0,
          transition: { delay: 0.1, duration: 0.5 }
        },
        hover: {
          y: -4,
          transition: { duration: 0.2 }
        }
      }}
    >
      {/* Spotlight Effect */}
      <motion.div
        className="pointer-events-none absolute -inset-px z-5 rounded-[inherit] opacity-0 transition-opacity duration-300 group-hover/spotlight:opacity-100"
        style={{
          background: useMotionTemplate`
            radial-gradient(
              450px circle at ${mouseX}px ${mouseY}px,
              rgba(59, 130, 246, 0.12),
              transparent 80%
            )
          `,
        }}
      />

      {/* Background Image Layer */}
      <motion.div
        className="absolute inset-0 z-0"
        variants={{
          initial: { filter: "blur(3px)", opacity: 0.5, scale: 1.1 },
          visible: { filter: "blur(0px)", opacity: 0.5, scale: 1.1 },
          hover: {
            filter: "blur(0px)",
            opacity: 1,
            scale: 1,
            transition: { duration: 0.6, ease: [0.22, 1, 0.36, 1] }
          }
        }}
      >
        <img
          src={service.imageUrl as string}
          alt={service.title}
          // fill
          className="object-cover w-full h-full absolute inset-0"
          sizes="(max-width: 768px) 100vw, 400px"
        />
      </motion.div>

      {/* Gradient Overlay for Text Readability on Hover */}
      <div className="absolute inset-0 z-10 pointer-events-none bg-gradient-to-t from-slate-950 via-slate-950/80 to-transparent opacity-80 transition-opacity duration-500 group-hover:opacity-60" />

      {/* Inner Glow Effect */}
      <motion.div
        className="pointer-events-none absolute inset-0 z-15 rounded-[inherit] opacity-0 transition-opacity duration-300 group-hover/spotlight:opacity-100"
        style={{
          background: useMotionTemplate`
            radial-gradient(
              300px circle at ${mouseX}px ${mouseY}px,
              rgba(59, 130, 246, 0.1),
              transparent 60%
            )
          `,
        }}
      />

      <div className="relative z-30 flex h-full flex-col justify-between p-8 pointer-events-none">
        <div className="flex w-full items-start justify-between">
          {/* Index Number */}
          <span className="font-mono text-6xl font-bold tracking-tighter text-slate-300 transition-colors duration-500 group-hover:text-slate-300/30">
            #{String(index + 1).padStart(2, "0")}
          </span>

          <div className="flex h-12 w-12 items-center justify-center rounded-xl bg-slate-800/50 text-slate-400 backdrop-blur-md transition-all duration-300 group-hover:bg-blue-500/20 group-hover:text-blue-400 group-hover:scale-110 pointer-events-auto">
            <Icon className="h-6 w-6" />
          </div>
        </div>

        <div>
          <h3 className="mb-2 text-2xl font-bold tracking-tight text-white transition-transform duration-300 group-hover:translate-x-1">
            {service.title}
          </h3>

          <div className="mt-4 flex items-center gap-2 text-sm font-medium text-blue-400 opacity-60 transition-all duration-300 group-hover:opacity-100 group-hover:translate-x-1 pointer-events-auto">
            <span>View Details</span>
            <ArrowRight className="h-4 w-4" />
          </div>
        </div>
      </div>
    </motion.div>
  );
}

function ServiceModal({
  service,
  icon: Icon,
  onClose,
  offeringsTitle,
}: {
  service: ServiceItem;
  index: number;
  icon: LucideIcon;
  onClose: () => void;
  offeringsTitle: string;
}) {
  // Lock body scroll
  useEffect(() => {
    document.body.style.overflow = "hidden";
    return () => {
      document.body.style.overflow = "";
    };
  }, []);

  return (
    <div className="fixed inset-0 z-50 flex items-center justify-center p-4 sm:p-6">
      <motion.div
        className="absolute inset-0 bg-slate-950/80 backdrop-blur-md"
        initial={{ opacity: 0 }}
        animate={{ opacity: 1 }}
        exit={{ opacity: 0 }}
        transition={{ duration: 0.2 }}
        onClick={onClose}
      />
      <motion.div
        className="relative w-full max-w-2xl overflow-hidden rounded-2xl border border-slate-700 bg-slate-900 shadow-2xl"
        initial={{ opacity: 0, scale: 0.95, y: 10 }}
        animate={{ opacity: 1, scale: 1, y: 0 }}
        exit={{ opacity: 0, scale: 0.95, y: 10 }}
        transition={{
          type: "spring",
          stiffness: 350,
          damping: 25,
          mass: 0.8,
        }}
        style={{ willChange: "transform, opacity" }}
      >
        <button
          onClick={onClose}
          className="absolute right-4 top-4 z-20 rounded-full bg-slate-800/50 p-2 text-slate-400 hover:bg-slate-700 hover:text-white transition-colors"
        >
          <X className="h-5 w-5" />
        </button>

        <div className="relative p-6 sm:p-10">
          <div className="pointer-events-none absolute -right-20 -top-20 h-[300px] w-[300px] rounded-full bg-blue-500/20 blur-[100px]" />
          <div className="pointer-events-none absolute -left-20 -bottom-20 h-[200px] w-[200px] rounded-full bg-emerald-500/10 blur-[80px]" />

          <div className="relative z-10 flex flex-col gap-6">
            <div className="flex items-start gap-4">
              <div className="flex h-14 w-14 shrink-0 items-center justify-center rounded-xl bg-blue-500/10 text-blue-400 ring-1 ring-blue-500/25">
                <Icon className="h-7 w-7" />
              </div>
              <div>
                <h3 className="text-2xl font-bold text-white md:text-3xl">
                  {service.title}
                </h3>
                <div className="mt-4 text-base leading-relaxed text-slate-300 md:text-lg">
                  {service.description}
                </div>
              </div>
            </div>

            <div className="mt-4 rounded-xl border border-slate-800/50 bg-slate-950/30 p-6">
              <h4 className="mb-4 flex items-center gap-2 text-sm font-semibold uppercase tracking-wider text-blue-400">
                <BarChart3 className="h-4 w-4" />
                {offeringsTitle}
              </h4>
              <ul className="grid gap-3 sm:grid-cols-2">
                {service.offerings.map((item, i) => (
                  <motion.li
                    key={i}
                    initial={{ opacity: 0, x: -10 }}
                    animate={{ opacity: 1, x: 0 }}
                    transition={{ delay: 0.1 + i * 0.05 }}
                    className="flex items-start gap-3 text-sm text-slate-300"
                  >
                    <Check className="mt-0.5 h-4 w-4 shrink-0 text-emerald-400" />
                    <span className="leading-snug">{item}</span>
                  </motion.li>
                ))}
              </ul>
            </div>
          </div>
        </div>
      </motion.div>
    </div>
  );
}

// --- Main Component ---

export interface ServicesProps {
  apiServices?: StrapiService[];
}

export function Services({ apiServices = [] }: ServicesProps) {
  const t = useTranslations("services");
  const targetRef = useRef<HTMLDivElement>(null);
  const { scrollYProgress } = useScroll({
    target: targetRef,
  });

  const services = useMemo((): ServiceItem[] => {
    if (apiServices.length > 0) {
      return apiServices.map(strapiServiceToItem);
    }
    const raw = t.raw("items");
    return isServiceItems(raw) ? raw : [];
  }, [apiServices, t]);

  // Transform for horizontal scroll
  const x = useTransform(scrollYProgress, [0, 1], ["1%", "-92%"]);

  const [selectedIdx, setSelectedIdx] = useState<number | null>(null);

  return (
    <section id="services" className="relative bg-slate-950">
      {/* Top glow */}
      <div className="absolute inset-x-0 top-0 h-px bg-linear-to-r from-transparent via-blue-500/50 to-transparent z-20" />

      {/* Ambient background */}
      <div className="fixed inset-0 overflow-hidden pointer-events-none z-0">
        <div className="absolute top-[20%] left-[10%] w-[500px] h-[500px] bg-blue-600/5 rounded-full blur-[120px]" />
        <div className="absolute bottom-[10%] right-[5%] w-[400px] h-[400px] bg-indigo-600/5 rounded-full blur-[100px]" />
      </div>

      {/* Mobile View */}
      <div className="block lg:hidden py-24 px-4 container relative z-10">
        <div className="mx-auto max-w-3xl text-center mb-12">
          <h2 className="text-3xl font-bold tracking-tight text-white">
            {t("title")}
          </h2>
          <p className="mt-4 text-lg text-slate-400">{t("subtitle")}</p>
        </div>
        <div className="grid grid-cols-1 gap-6 sm:grid-cols-2">
          {services.map((svc, i) => (
            <ServiceCard
              key={i}
              index={i}
              service={svc}
              icon={serviceIcons[i] ?? Code2}
              onClick={() => setSelectedIdx(i)}
              className="h-[320px]"
            />
          ))}
        </div>
      </div>

      {/* Desktop View */}
      <div ref={targetRef} className="hidden lg:block relative h-[700vh] z-10">
        <div className="sticky top-0 flex h-screen items-center overflow-hidden">
          <div className=" top-12 left-0 right-0 z-20 text-center pointer-events-none ps-36">
            <motion.div
              style={{ opacity: useTransform(scrollYProgress, [0, 0.1], [1, 0]) }}
            >
              <h2 className="text-4xl text-start text-nowrap font-bold tracking-tight text-white md:text-5xl lg:leading-tight">
                {t("title")}
              </h2>
              <p className="mt-4 text-xl text-start text-slate-400 min-w-120">
                {t("subtitle")}
              </p>
            </motion.div>
          </div>

          <motion.div style={{ x }} className="flex gap-10 px-24 items-center relative z-30">
            <div className="w-[10vw] flex-shrink-0" />

            {services.map((svc, i) => (
              <ServiceCard
                key={i}
                index={i}
                service={svc}
                icon={serviceIcons[i] ?? Code2}
                onClick={() => setSelectedIdx(i)}
                className="w-[400px] h-[550px] relative z-30"
              />
            ))}
            <div className="w-[10vw] flex-shrink-0" />
          </motion.div>
          <ScrollRotate
            speed={0.15}
            className="absolute top-12 left-12 -z-10"
          >
            <Image
              src="/globe.png"
              alt="Services"
              className="aspect-square blur-[1px] opacity-50"
              height={200}
              width={200}
            />
          </ScrollRotate>
        </div>
      </div>

      <AnimatePresence>
        {selectedIdx !== null && services[selectedIdx] && (
          <ServiceModal
            service={services[selectedIdx]}
            index={selectedIdx}
            icon={serviceIcons[selectedIdx] ?? Code2}
            offeringsTitle={t("offeringsTitle")}
            onClose={() => setSelectedIdx(null)}
          />
        )}
      </AnimatePresence>
    </section>
  );
}
