"use client";

import { useMemo, useCallback } from "react";
import { useTranslations } from "next-intl";
import Link from "next/link";
import { motion, useMotionValue, useMotionTemplate } from "motion/react";
import { cn } from "@/core/lib/utils";
import { ArrowUpRight } from "lucide-react";

type SolutionItem = { label: string; link: string };

function isSolutionItems(value: unknown): value is SolutionItem[] {
  if (!Array.isArray(value)) return false;
  return value.every((item) => {
    if (!item || typeof item !== "object") return false;
    const asAny = item as Record<string, unknown>;
    return typeof asAny.label === "string" && typeof asAny.link === "string";
  });
}

function SolutionCard({
  item,
  index,
}: {
  item: SolutionItem;
  index: number;
}) {
  const mouseX = useMotionValue(0);
  const mouseY = useMotionValue(0);

  const handleMouseMove = useCallback(
    (event: React.MouseEvent<HTMLDivElement>) => {
      const { currentTarget, clientX, clientY } = event;
      const { left, top } = currentTarget.getBoundingClientRect();
      mouseX.set(clientX - left);
      mouseY.set(clientY - top);
    },
    [mouseX, mouseY]
  );

  return (
    <motion.div
      initial="initial"
      whileInView="visible"
      viewport={{ once: true, amount: 0.2 }}
      variants={{
        initial: { opacity: 0, y: 25, filter: "blur(12px)" },
        visible: {
          opacity: 1,
          y: 0,
          filter: "blur(0px)",
          transition: { delay: index * 0.05, duration: 0.5, ease: [0.22, 1, 0.36, 1] },
        },
      }}
      whileHover={{ y: -4, scale: 1.02, transition: { duration: 0.2 } }}
      className="group/spotlight group"
      onMouseMove={handleMouseMove}
    >
      <Link
        href={item.link}
        className={cn(
          "relative flex items-center justify-between gap-3 overflow-hidden rounded-2xl border border-slate-700/80 bg-slate-900/60 px-5 py-4 backdrop-blur-sm",
          "transition-all duration-300 hover:border-blue-500/50 hover:bg-slate-800/70",
          "hover:shadow-[0_0_28px_-8px_rgba(59,130,246,0.35)]"
        )}
      >
        {/* Spotlight Effect */}
        <motion.div
          className="pointer-events-none absolute -inset-px z-0 rounded-[inherit] opacity-0 transition-opacity duration-300 group-hover/spotlight:opacity-100"
          style={{
            background: useMotionTemplate`
              radial-gradient(
                250px circle at ${mouseX}px ${mouseY}px,
                rgba(59, 130, 246, 0.15),
                transparent 80%
              )
            `,
          }}
        />
        <span className="relative z-10 text-sm font-medium text-slate-200 transition-colors group-hover:text-white sm:text-base">
          {item.label}
        </span>
        <span className="relative z-10 flex h-8 w-8 shrink-0 items-center justify-center rounded-lg bg-slate-700/60 text-slate-400 transition-all duration-300 group-hover:bg-blue-500/20 group-hover:text-blue-400">
          <ArrowUpRight className="h-4 w-4" />
        </span>
      </Link>
    </motion.div>
  );
}

export function Solutions() {
  const t = useTranslations("solutions");

  const solutions = useMemo(() => {
    const raw = t.raw("solutions");
    return isSolutionItems(raw) ? raw : [];
  }, [t]);

  const sectionId = t("section_id");

  return (
    <section
      id={sectionId}
      className="relative w-full overflow-hidden bg-slate-950 py-24 md:py-32"
    >
      {/* Top edge glow */}
      <div className="absolute inset-x-0 top-0 z-10 h-px bg-linear-to-r from-transparent via-blue-500/50 to-transparent" />

      {/* Dot-grid / particles-like background */}
      <div
        className="pointer-events-none absolute inset-0 z-0 opacity-[0.15]"
        aria-hidden
      >
        <div
          className="h-full w-full"
          style={{
            backgroundImage: `radial-gradient(circle at 1px 1px, rgb(255 255 255) 1px, transparent 0)`,
            backgroundSize: "32px 32px",
          }}
        />
      </div>

      <div className="container relative z-10">
        <motion.div
          className="mx-auto max-w-3xl text-center"
          initial={{ opacity: 0, y: 30, filter: "blur(15px)" }}
          whileInView={{ opacity: 1, y: 0, filter: "blur(0px)" }}
          viewport={{ once: true, amount: 0.3 }}
          transition={{ duration: 0.8, ease: [0.22, 1, 0.36, 1] }}
        >
          <motion.h2
            className="text-3xl font-bold tracking-tight text-white md:text-4xl lg:text-5xl"
            initial={{ opacity: 0, y: 20, filter: "blur(12px)" }}
            whileInView={{ opacity: 1, y: 0, filter: "blur(0px)" }}
            viewport={{ once: true }}
            transition={{ duration: 0.6, delay: 0.1 }}
          >
            {t("title")}
          </motion.h2>
          <motion.p
            className="mt-4 text-lg text-slate-400 md:text-xl"
            initial={{ opacity: 0, y: 15, filter: "blur(10px)" }}
            whileInView={{ opacity: 1, y: 0, filter: "blur(0px)" }}
            viewport={{ once: true }}
            transition={{ duration: 0.6, delay: 0.2 }}
          >
            {t("description")}
          </motion.p>
        </motion.div>

        <div className="mx-auto mt-14 grid max-w-5xl grid-cols-1 gap-4 sm:grid-cols-2 lg:grid-cols-3">
          {solutions.map((item, index) => (
            <SolutionCard key={`${item.label}-${index}`} item={item} index={index} />
          ))}
        </div>
      </div>
    </section>
  );
}
