"use client";

import { useTranslations } from "next-intl";
import { AnimatedTestimonials } from "@/components/ui/animated-testimonials";
import { motion, type Variants, type Easing } from "motion/react";

// Custom easing
const customEase: Easing = [0.22, 1, 0.36, 1];

const containerVariants: Variants = {
  hidden: { opacity: 0 },
  visible: {
    opacity: 1,
    transition: {
      staggerChildren: 0.2,
      delayChildren: 0.1,
    },
  },
};

const headerVariants: Variants = {
  hidden: {
    opacity: 0,
    y: 40,
    filter: "blur(15px)"
  },
  visible: {
    opacity: 1,
    y: 0,
    filter: "blur(0px)",
    transition: {
      duration: 0.8,
      ease: customEase,
    },
  },
};

const contentVariants: Variants = {
  hidden: {
    opacity: 0,
    y: 30,
    filter: "blur(20px)",
    scale: 0.95,
  },
  visible: {
    opacity: 1,
    y: 0,
    filter: "blur(0px)",
    scale: 1,
    transition: {
      duration: 0.8,
      ease: customEase,
    },
  },
};

export function Testimonials() {
  const t = useTranslations("testimonials");
  const items = (t.raw("items") ?? []) as Array<{
    quote: string;
    name: string;
    designation: string;
    src: string;
  }>;

  return (
    <section id="testimonials" className="relative w-full overflow-hidden bg-slate-950 py-24 md:py-32">
      {/* Top edge glow with animation */}
      <motion.div
        className="absolute inset-x-0 top-0 z-10 h-px bg-linear-to-r from-transparent via-blue-500/50 to-transparent"
        initial={{ scaleX: 0, opacity: 0 }}
        whileInView={{ scaleX: 1, opacity: 1 }}
        viewport={{ once: true }}
        transition={{ duration: 1.2, ease: "easeOut" }}
      />

      {/* Background pattern */}
      <div
        className="pointer-events-none absolute inset-0 z-0 opacity-[0.15]"
        aria-hidden
      >
        <div
          className="h-full w-full"
          style={{
            backgroundImage: `radial-gradient(circle at 1px 1px, rgb(255 255 255) 1px, transparent 0)`,
            backgroundSize: "32px 32px",
          }}
        />
      </div>

      {/* Ambient glow */}
      <motion.div
        className="absolute top-1/4 left-1/4 h-96 w-96 bg-blue-500/10 rounded-full blur-[150px] pointer-events-none"
        animate={{
          scale: [1, 1.2, 1],
          opacity: [0.3, 0.5, 0.3],
        }}
        transition={{
          duration: 8,
          repeat: Infinity,
          ease: "easeInOut",
        }}
      />
      <motion.div
        className="absolute bottom-1/4 right-1/4 h-80 w-80 bg-purple-500/10 rounded-full blur-[120px] pointer-events-none"
        animate={{
          scale: [1.2, 1, 1.2],
          opacity: [0.3, 0.5, 0.3],
        }}
        transition={{
          duration: 8,
          repeat: Infinity,
          ease: "easeInOut",
          delay: 2,
        }}
      />

      <div className="container relative z-10">
        <motion.div
          className="mx-auto flex max-w-[980px] flex-col items-center gap-4 text-center mb-12"
          variants={containerVariants}
          initial="hidden"
          whileInView="visible"
          viewport={{ once: true, amount: 0.3 }}
        >
          <motion.h2
            className="text-3xl font-bold tracking-tight text-white md:text-4xl lg:text-5xl"
            variants={headerVariants}
          >
            {t("title")}
          </motion.h2>
          <motion.p
            className="text-lg text-slate-400 md:text-xl"
            variants={headerVariants}
          >
            {t("subtitle")}
          </motion.p>
        </motion.div>

        {/* Force dark mode for the component since we are on a dark background */}
        <motion.div
          className="dark"
          variants={contentVariants}
          initial="hidden"
          whileInView="visible"
          viewport={{ once: true, amount: 0.2 }}
        >
          <AnimatedTestimonials testimonials={items} autoplay />
        </motion.div>
      </div>
    </section>
  );
}
