"use client";

import { useRef, useEffect, useMemo, useCallback } from "react";
import {
  motion,
  useScroll,
  useTransform,
  useSpring,
  useReducedMotion,
  useMotionValueEvent,
} from "framer-motion";
import { useTranslations } from "next-intl";
import { Mail, ChevronDown, Mouse } from "lucide-react";

import { MotionButton } from "@/components/ui/button";
import { BackgroundBeams } from "@/components/ui/background-beams";
import { SparklesCanvas } from "@/components/ui/sparkles";

// --- Constants ---
const TOTAL_FRAMES = 273;

// --- Helper: Frame Path ---
function getFramePath(frameNum: number): string {
  // Ensure the path matches your public folder structure exactly
  return `/hero-frames/ezgif-frame-${String(frameNum).padStart(3, "0")}.jpg`;
}

// --- Particle Data (Static) ---
const particleData = [
  { left: 15, top: 20, duration: 3.5, delay: 0.2 },
  { left: 85, top: 15, duration: 4.2, delay: 0.8 },
  { left: 45, top: 80, duration: 3.8, delay: 1.2 },
  { left: 70, top: 45, duration: 4.5, delay: 0.5 },
  { left: 25, top: 60, duration: 3.2, delay: 1.5 },
  { left: 90, top: 70, duration: 4.0, delay: 0.3 },
  { left: 10, top: 85, duration: 3.6, delay: 1.0 },
  { left: 55, top: 25, duration: 4.3, delay: 0.7 },
  { left: 35, top: 90, duration: 3.4, delay: 1.8 },
  { left: 75, top: 55, duration: 4.1, delay: 0.4 },
  { left: 5, top: 40, duration: 3.9, delay: 1.3 },
  { left: 60, top: 10, duration: 4.4, delay: 0.6 },
  { left: 40, top: 65, duration: 3.3, delay: 1.6 },
  { left: 80, top: 30, duration: 4.2, delay: 0.9 },
  { left: 20, top: 75, duration: 3.7, delay: 1.1 },
  { left: 95, top: 50, duration: 4.0, delay: 0.1 },
  { left: 30, top: 35, duration: 3.5, delay: 1.4 },
  { left: 65, top: 85, duration: 4.3, delay: 0.2 },
  { left: 50, top: 5, duration: 3.8, delay: 1.7 },
  { left: 12, top: 55, duration: 4.1, delay: 0.8 },
];

// --- Sub-component: Floating Particles ---
// Defined outside or memoized to ensure strict isolation
const FloatingParticles = () => {
  return (
    <div className="absolute inset-0 z-10 pointer-events-none">
      {particleData.map((particle, i) => (
        <motion.div
          key={i}
          className="absolute w-1 h-1 bg-blue-500/30 rounded-full"
          style={{
            left: `${particle.left}%`,
            top: `${particle.top}%`,
          }}
          animate={{
            y: [0, -30, 0],
            opacity: [0.3, 0.8, 0.3],
            scale: [1, 1.5, 1],
          }}
          transition={{
            duration: particle.duration,
            repeat: Infinity,
            delay: particle.delay,
            ease: "easeInOut",
          }}
        />
      ))}
    </div>
  );
};

// --- Main Component ---

const VideoHero = () => {
  const t = useTranslations("hero");
  const sectionRef = useRef<HTMLElement>(null);
  const canvasRef = useRef<HTMLCanvasElement>(null);

  // Refs for logic that doesn't need to trigger re-renders
  const contextRef = useRef<CanvasRenderingContext2D | null>(null);
  const imagesRef = useRef<HTMLImageElement[]>([]);
  const frameLoadStatusRef = useRef<boolean[]>([]);

  const prefersReducedMotion = useReducedMotion();

  // 1. Scroll Hooks
  const { scrollYProgress } = useScroll({
    target: sectionRef,
    offset: ["start start", "end start"],
  });

  const smoothProgress = useSpring(scrollYProgress, {
    mass: 0.2,      // Increased mass for weightier feel
    stiffness: 150, // High stiffness to catch up quickly
    damping: 30,    // No bounce
    restDelta: 0.001,
  });

  // 2. Transformations for Text/UI (Standard Framer Motion)
  const introOpacity = useTransform(smoothProgress, [0, 0.15], [0.8, 1]);
  const introScale = useTransform(smoothProgress, [0, 0.25], [prefersReducedMotion ? 1 : 0.98, 1]);
  const contentY = useTransform(smoothProgress, [0.25, 0.8], [0, -80]);
  const contentOpacity = useTransform(smoothProgress, [0.7, 0.9], [1, 0]);
  const secondaryY = useTransform(smoothProgress, [0.25, 0.8], [0, -40]);
  const indicatorOpacity = useTransform(smoothProgress, [0, 0.1], [1, 0]); // Fade out scroll indicator quickly

  // Headline Animations
  const headline1Opacity = useTransform(smoothProgress, [0.4, 0.5], [1, 0]);
  const headline1Rotate = useTransform(smoothProgress, [0.4, 0.5], [0, 90]);
  const headline1Y = useTransform(smoothProgress, [0.4, 0.5], [0, -50]);

  const headline2Opacity = useTransform(smoothProgress, [0.5, 0.6], [0, 1]);
  const headline2Rotate = useTransform(smoothProgress, [0.5, 0.6], [-90, 0]);
  const headline2Y = useTransform(smoothProgress, [0.5, 0.6], [50, 0]);

  // 3. Canvas Rendering Logic (The "Cover" function)
  const renderFrame = useCallback((index: number) => {
    const canvas = canvasRef.current;
    const ctx = contextRef.current;
    const img = imagesRef.current[index];

    if (!canvas || !ctx || !img) return;

    // Only draw if image is actually loaded
    if (!frameLoadStatusRef.current[index]) return;

    // Calculate "object-fit: cover" dimensions
    const canvasWidth = canvas.width;
    const canvasHeight = canvas.height;
    const imgWidth = img.width;
    const imgHeight = img.height;

    const ratio = Math.max(canvasWidth / imgWidth, canvasHeight / imgHeight);
    const centerShift_x = (canvasWidth - imgWidth * ratio) / 2;
    const centerShift_y = (canvasHeight - imgHeight * ratio) / 2;

    ctx.clearRect(0, 0, canvasWidth, canvasHeight);
    ctx.drawImage(
      img,
      0,
      0,
      imgWidth,
      imgHeight,
      centerShift_x,
      centerShift_y,
      imgWidth * ratio,
      imgHeight * ratio
    );
  }, []);

  // 4. Setup: Preload Images & Init Canvas
  useEffect(() => {
    const canvas = canvasRef.current;
    if (!canvas) return;

    // Set Context
    contextRef.current = canvas.getContext("2d", { alpha: false }); // alpha: false for performance

    // Handle High DPI (Retina) Screens for Sharpness
    const handleResize = () => {
      if (!canvas) return;
      const dpr = window.devicePixelRatio || 1;
      // Set internal buffer size to match screen pixels
      canvas.width = window.innerWidth * dpr;
      canvas.height = window.innerHeight * dpr;

      // Scale context ensures drawing operations use CSS pixels logically
      // Note: For drawImage with specific pixel manipulation, we often skip context.scale 
      // and just calculate dimensions based on canvas.width/height as done in renderFrame above.

      // Re-render current frame after resize
      const currentScroll = smoothProgress.get();
      const frameIndex = Math.max(1, Math.min(TOTAL_FRAMES, Math.round(currentScroll * (TOTAL_FRAMES - 1)) + 1));
      renderFrame(frameIndex);
    };

    window.addEventListener('resize', handleResize);
    handleResize(); // Init size

    // Start Preloading
    const preloadImages = () => {
      for (let i = 1; i <= TOTAL_FRAMES; i++) {
        const img = new window.Image();
        img.src = getFramePath(i);
        img.onload = () => {
          frameLoadStatusRef.current[i] = true;
          // If this is the very first frame, render it immediately
          if (i === 1) renderFrame(1);
        };
        imagesRef.current[i] = img;
      }
    };
    preloadImages();

    return () => window.removeEventListener('resize', handleResize);
  }, [renderFrame, smoothProgress]);

  // 5. The Loop: Update Canvas on Scroll
  useMotionValueEvent(smoothProgress, "change", (latest) => {
    // Calculate frame index (1 to TOTAL_FRAMES)
    const frameIndex = Math.max(1, Math.min(TOTAL_FRAMES, Math.round(latest * (TOTAL_FRAMES - 1)) + 1));

    // Draw directly to canvas (No React State Update!)
    renderFrame(frameIndex);
  });

  return (
    <motion.section
      ref={sectionRef}
      className="relative w-full bg-slate-950"
      style={{ height: "350vh" }}
    >
      <div className="sticky top-0 h-screen w-full overflow-hidden flex flex-col items-center justify-center">

        {/* --- CANVAS LAYER --- */}
        {/* We use basic CSS to stretch the canvas, but internal resolution is set in JS */}
        <canvas
          ref={canvasRef}
          className="absolute inset-0 z-0 w-full h-full object-cover"
          style={{
            pointerEvents: 'none'
          }}
        />

        {/* --- OVERLAYS (Static, no re-renders) --- */}
        <div className="absolute inset-0 bg-slate-950/60 z-[5] pointer-events-none" />

        {/* Memoized components or static JSX won't re-render because we removed state updates */}
        <FloatingParticles />
        <div className="absolute inset-0 z-10 pointer-events-none">
          <BackgroundBeams className="opacity-40" />
        </div>

        {/* --- CONTENT LAYER --- */}
        <motion.div
          className="relative z-20 container flex flex-col items-center text-center px-4"
          style={{
            opacity: introOpacity,
            scale: introScale,
          }}
        >
          <motion.div style={{ y: contentY, opacity: contentOpacity }}>
            <motion.span
              initial={{ opacity: 0, scale: 0.8 }}
              animate={{ opacity: 1, scale: 1 }}
              transition={{ duration: 0.5 }}
              className="inline-block px-4 py-2 mb-8 text-sm font-semibold text-blue-400 bg-blue-500/10 border border-blue-500/20 rounded-full"
            >
            {t("badge")}
            </motion.span>

            {/* Headlines */}
            <div className="h-32 perspective-1000 relative flex items-center justify-center mb-4">
              <motion.div
                className="absolute inset-0 flex flex-col lg:flex-row items-center justify-center gap-4 text-5xl md:text-7xl lg:text-8xl font-bold text-white whitespace-nowrap"
                style={{
                  opacity: headline1Opacity,
                  rotateX: headline1Rotate,
                  y: headline1Y,
                  transformStyle: "preserve-3d",
                }}
              >
                <span>{t("flip1.prefix")}</span>
                <span className="text-transparent bg-clip-text bg-gradient-to-r from-secondary via-secondary/80 to-secondary">
                  {t("flip1.highlight")}
                </span>
              </motion.div>

              <motion.div
                className="absolute inset-0 flex flex-col lg:flex-row items-center justify-center gap-4 text-5xl md:text-7xl lg:text-8xl font-bold text-white whitespace-nowrap"
                style={{
                  opacity: headline2Opacity,
                  rotateX: headline2Rotate,
                  y: headline2Y,
                  transformStyle: "preserve-3d",
                }}
              >
                <span>{t("flip2.prefix")}</span>
                <span className="text-transparent bg-clip-text bg-gradient-to-r from-secondary via-secondary/80 to-secondary">
                  {t("flip2.highlight")}
                </span>
              </motion.div>
            </div>

            <div className="overflow-hidden w-120 absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 pointer-events-none">
              <SparklesCanvas
                className="w-full max-w-120 mx-auto mb-8 h-20"
                particleColor="var(--secondary)"
                particleDensity={60}
                minSize={0.5}
                maxSize={1.5}
                speed={0.8}
              />
            </div>

            <motion.div style={{ y: secondaryY }}>
              <p className="text-xl md:text-2xl text-slate-400 max-w-2xl mx-auto mb-12">
                {t("subtitle")}
              </p>

              <div className="flex justify-center gap-4 z-20 pointer-events-auto">
                <MotionButton
                  variant="gradient-outline"
                  size="xl"
                  iconLeft={<Mail className="w-6 h-6" />}
                  className="px-8 py-6"
                  whileHover={{ scale: 1.05 }}
                  whileTap={{ scale: 0.95 }}
                  animate={{
                    boxShadow: [
                      "0 0 20px rgba(6, 182, 212, 0.3)",
                      "0 0 40px rgba(6, 182, 212, 0.5)",
                      "0 0 20px rgba(6, 182, 212, 0.3)",
                    ],
                  }}
                  transition={{
                    boxShadow: {
                      duration: 2,
                      repeat: Infinity,
                      ease: "easeInOut",
                    },
                  }}
                >
                  {t("cta")}
                </MotionButton>
              </div>
            </motion.div>
          </motion.div>
        </motion.div>

        {/* Scroll Indicator */}
        <motion.div
          style={{ opacity: indicatorOpacity }}
          className="absolute bottom-10 z-30 pointer-events-auto"
        >
          <div
            className="flex flex-col items-center gap-2 text-slate-400 cursor-pointer hover:text-white transition-colors"
            onClick={() =>
              document.getElementById("services")?.scrollIntoView({ behavior: "smooth" })
            }
          >
            <Mouse className="w-6 h-6" />
            <ChevronDown className="w-5 h-5 animate-bounce" />
          </div>
        </motion.div>
      </div>
    </motion.section>
  );
};

export default VideoHero;