"use client";

import { useTranslations } from "next-intl";
import { useRef, useEffect, useState } from "react";
import Image from "next/image";
import { motion, useInView, animate, useScroll, useMotionValue, useMotionTemplate } from "motion/react";
import { Users, Clock, FolderKanban, Globe, Check, Handshake, Leaf } from "lucide-react";
import Lottie, { LottieRefCurrentProps } from "lottie-react";
import { ScrollRotate } from "@/components/ui/scroll-rotate";

function AnimatedCounter({
  value,
  suffix = "",
  duration = 1.4,
}: {
  value: number;
  suffix?: string;
  duration?: number;
}) {
  const ref = useRef<HTMLSpanElement>(null);
  const isInView = useInView(ref, { once: true, amount: 0.3 });
  const [display, setDisplay] = useState(0);

  useEffect(() => {
    if (!isInView) return;
    const controls = animate(0, value, {
      duration,
      ease: [0.22, 1, 0.36, 1],
      onUpdate: (v) => setDisplay(Math.round(v)),
    });
    return () => controls.stop();
  }, [isInView, value, duration]);

  return (
    <span ref={ref}>
      {display}
      {suffix}
    </span>
  );
}

// Separate component for scroll-pinned Lottie animation
// This ensures useScroll only runs when the component is mounted
function ScrollPinnedLottie({ animationData }: { animationData: object }) {
  const t = useTranslations("whyus");
  const lottieRef = useRef<LottieRefCurrentProps>(null);
  const scrollContainerRef = useRef<HTMLDivElement>(null);

  const { scrollYProgress } = useScroll({
    target: scrollContainerRef,
    offset: ["start start", "end end"],
  });

  useEffect(() => {
    const unsubscribe = scrollYProgress.on("change", (progress) => {
      if (lottieRef.current) {
        const totalFrames = lottieRef.current.getDuration(true) || 100;
        const frame = Math.floor(progress * totalFrames);
        lottieRef.current.goToAndStop(frame, true);
      }
    });

    return () => unsubscribe();
  }, [scrollYProgress]);

  return (
    <div
      ref={scrollContainerRef}
      className="relative bg-slate-950"
      style={{ height: "500vh" }}
    >
      <div className="sticky top-0 flex h-screen w-full items-center justify-center overflow-hidden">
        {/* Background gradient */}
        <div className="absolute inset-0 bg-linear-to-b  " />

        {/* Animated background elements */}
        <motion.div
          className="absolute inset-0 opacity-30"
          style={{
            background:
              "radial-gradient(circle at 50% 50%, rgba(59, 130, 246, 0.15) 0%, transparent 50%)",
          }}
        />

        <ScrollRotate
          speed={0.15}
          className="absolute top-12 right-12 -z-10"
        >
          <Image
            src="/shape-globe-2.png"
            alt="Services"
            className="aspect-square blur-[1px] opacity-50"
            height={200}
            width={200}
          />
        </ScrollRotate>
        <ScrollRotate
          speed={0.15}
          className="absolute bottom-12 left-12 -z-10"
        >
          <Image
            src="/globe.png"
            alt="Services"
            className="aspect-square blur-[1px] opacity-50"
            height={200}
            width={200}
          />
        </ScrollRotate>

        {/* Lottie Animation */}
        <motion.div
          className="relative z-10 flex h-full w-full items-center justify-center"
          initial={{ opacity: 0, scale: 0.9 }}
          whileInView={{ opacity: 1, scale: 1 }}
          viewport={{ once: true }}
          transition={{ duration: 0.8, ease: [0.22, 1, 0.36, 1] }}
        >
          <div className="h-[60vh] w-auto max-w-[90vw] md:h-[70vh] lg:h-[80vh]">
            <Lottie
              lottieRef={lottieRef}
              animationData={animationData}
              autoplay={false}
              loop={false}
              className="h-full w-full"
            />
          </div>
        </motion.div>

        {/* Scroll indicator */}
        <motion.div
          className="absolute bottom-8 left-1/2 flex -translate-x-1/2 flex-col items-center gap-2"
          initial={{ opacity: 0, y: -10 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ delay: 1, duration: 0.5 }}
        >
          <span className="text-xs font-medium uppercase tracking-widest text-slate-500">
            {t("scrollToAnimate")}
          </span>
          <motion.div
            className="h-12 w-6 rounded-full border-2 border-slate-600 p-1"
            initial={{ opacity: 0.5 }}
          >
            <motion.div
              className="h-2 w-2 rounded-full bg-blue-500"
              animate={{ y: [0, 16, 0] }}
              transition={{ duration: 1.5, repeat: Infinity, ease: "easeInOut" }}
            />
          </motion.div>
        </motion.div>
      </div>
    </div>
  );
}

const statIcons = [Users, Clock, FolderKanban, Globe];

const principleCardVariants = {
  hidden: { opacity: 0, y: 50, rotateX: -15 },
  visible: {
    opacity: 1,
    y: 0,
    rotateX: 0,
    transition: { duration: 0.8, ease: [0.22, 1, 0.36, 1] as [number, number, number, number] },
  },
} as const;

function PrincipleCard({
  title,
  description,
  icon: Icon,
}: {
  title: string;
  description: string;
  icon: React.ComponentType<{ className?: string }>;
}) {
  const mouseX = useMotionValue(0);
  const mouseY = useMotionValue(0);

  const handleMouseMove = (event: React.MouseEvent<HTMLDivElement>) => {
    const { currentTarget, clientX, clientY } = event;
    const { left, top } = currentTarget.getBoundingClientRect();
    mouseX.set(clientX - left);
    mouseY.set(clientY - top);
  };

  return (
    <motion.div
      variants={principleCardVariants}
      whileHover={{ scale: 1.02, y: -5, transition: { duration: 0.2 } }}
      whileTap={{ scale: 0.98 }}
      onMouseMove={handleMouseMove}
      className="group/spotlight group relative h-full rounded-xl overflow-hidden border border-slate-700 bg-slate-900/50 backdrop-blur-sm hover:border-blue-500/50 transition-colors"
      style={{
        perspective: "1000px",
        willChange: "transform"
      }}
    >
      <motion.div
        className="pointer-events-none absolute -inset-px z-0 rounded-[inherit] opacity-0 transition-opacity duration-300 group-hover/spotlight:opacity-100"
        style={{
          background: useMotionTemplate`
            radial-gradient(
              300px circle at ${mouseX}px ${mouseY}px,
              rgba(59, 130, 246, 0.15),
              transparent 80%
            )
          `,
        }}
      />
      <motion.div
        className="pointer-events-none absolute inset-0 z-0 rounded-[inherit] opacity-0 transition-opacity duration-300 group-hover/spotlight:opacity-100"
        style={{
          background: useMotionTemplate`
            radial-gradient(
              180px circle at ${mouseX}px ${mouseY}px,
              rgba(59, 130, 246, 0.1),
              transparent 60%
            )
          `,
        }}
      />
      <div className="relative z-10 p-6 h-full flex flex-col">
        <motion.div className="mb-4">
          <Icon className="w-10 h-10 text-secondary" />
        </motion.div>
        <h3 className="text-xl font-semibold text-white mb-3">{title}</h3>
        <p className="text text-muted-foreground">{description}</p>
      </div>
    </motion.div>
  );
}

export function WhyUs() {
  const t = useTranslations("whyus");
  const heroT = useTranslations("hero");
  const [complexAnimationData, setComplexAnimationData] = useState<object | null>(null);

  useEffect(() => {
    fetch("/complex.json")
      .then((res) => res.json())
      .then(setComplexAnimationData)
      .catch(() => { });
  }, []);

  const stats = [
    { value: parseInt(t("stat1.value"), 10), suffix: t("stat1.suffix"), label: t("stat1.label") },
    { value: parseInt(t("stat2.value"), 10), suffix: t("stat2.suffix"), label: t("stat2.label") },
    { value: parseInt(t("stat3.value"), 10), suffix: t("stat3.suffix"), label: t("stat3.label") },
    { value: parseInt(t("stat4.value"), 10), suffix: t("stat4.suffix"), label: t("stat4.label") },
  ];

  const items = [t("item1"), t("item2"), t("item3"), t("item4"), t("item5")];

  return (
    <>
      {/* Scroll-Pinned Lottie Animation Section */}
      {complexAnimationData && (
        <ScrollPinnedLottie animationData={complexAnimationData} />
      )}

      <section
        id="whyus"
        className="relative w-full overflow-hidden bg-slate-950 py-24 md:py-32"
      >
        <div className="absolute inset-x-0 top-0 h-px bg-linear-to-r from-transparent via-blue-500/40 to-transparent" />

        <div className="container relative z-10">
          {/* Header */}
          <motion.div
            className="mx-auto max-w-3xl text-center"
            initial={{ opacity: 0, y: 30, filter: "blur(15px)" }}
            whileInView={{ opacity: 1, y: 0, filter: "blur(0px)" }}
            viewport={{ once: true, amount: 0.3 }}
            transition={{ duration: 0.8, ease: [0.22, 1, 0.36, 1] }}
          >
            <motion.p
              className="text-sm font-medium uppercase tracking-widest text-blue-400"
              initial={{ opacity: 0, y: 10, filter: "blur(8px)" }}
              whileInView={{ opacity: 1, y: 0, filter: "blur(0px)" }}
              viewport={{ once: true }}
              transition={{ duration: 0.5, delay: 0.1 }}
            >
              {t("subtitle")}
            </motion.p>
            <motion.h2
              className="mt-2 text-3xl font-bold tracking-tight text-white md:text-4xl lg:text-5xl"
              initial={{ opacity: 0, y: 20, filter: "blur(12px)" }}
              whileInView={{ opacity: 1, y: 0, filter: "blur(0px)" }}
              viewport={{ once: true }}
              transition={{ duration: 0.6, delay: 0.2 }}
            >
              {t("title")}
            </motion.h2>
          </motion.div>

          {/* Stats */}
          <div className="mx-auto mt-16 grid max-w-5xl grid-cols-2 gap-4 md:grid-cols-4 md:gap-6">
            {stats.map((stat, i) => {
              const Icon = statIcons[i];
              return (
                <motion.div
                  key={stat.label}
                  className="relative rounded-xl border border-slate-700/60 bg-slate-800/50 p-6 backdrop-blur-sm transition-colors hover:border-blue-500/30 hover:bg-slate-800/70"
                  initial={{ opacity: 0, y: 30, filter: "blur(15px)", scale: 0.95 }}
                  whileInView={{ opacity: 1, y: 0, filter: "blur(0px)", scale: 1 }}
                  viewport={{ once: true, amount: 0.2 }}
                  transition={{
                    duration: 0.6,
                    delay: i * 0.12,
                    ease: [0.22, 1, 0.36, 1],
                  }}
                  whileHover={{ y: -5, scale: 1.02 }}
                >
                  <div className="flex h-10 w-10 items-center justify-center rounded-lg bg-blue-500/15 text-blue-400">
                    <Icon className="h-5 w-5" />
                  </div>
                  <div className="mt-4 text-2xl font-bold tabular-nums text-white md:text-3xl">
                    <AnimatedCounter value={stat.value} suffix={stat.suffix} />
                  </div>
                  <p className="mt-1 text-sm text-slate-400">{stat.label}</p>
                </motion.div>
              );
            })}
          </div>

          <div className="mx-auto mt-20 grid max-w-6xl grid-cols-1 gap-12 lg:grid-cols-12 lg:gap-16">



            <div className="lg:col-span-7 space-y-12">
              <motion.div
                initial={{ opacity: 0, x: -20 }}
                whileInView={{ opacity: 1, x: 0 }}
                viewport={{ once: true, amount: 0.2 }}
                transition={{ duration: 0.6, ease: [0.22, 1, 0.36, 1] }}
              >
                <p className="text-lg leading-relaxed md:text-2xl  ">
                  {t("intro")}
                </p>
              </motion.div>
              <motion.ul
                className="space-y-4 lg:col-span-2"
                initial="hidden"
                whileInView="visible"
                viewport={{ once: true, amount: 0.15 }}
                variants={{
                  hidden: {},
                  visible: {
                    transition: { staggerChildren: 0.08, delayChildren: 0.1 },
                  },
                }}
              >
                {items.map((item, i) => (
                  <motion.li
                    key={i}
                    className="flex gap-3"
                    variants={{
                      hidden: { opacity: 0, x: 16 },
                      visible: {
                        opacity: 1,
                        x: 0,
                        transition: { ease: [0.22, 1, 0.36, 1] },
                      },
                    }}
                  >
                    <span className="mt-0.5 shrink-0 flex h-6 w-6 items-center justify-center rounded-full bg-secondary/20 text-secondary">
                      <Check className="h-3.5 w-3.5" strokeWidth={2.5} />
                    </span>
                    <span className="text-muted-foreground">{item}</span>
                  </motion.li>
                ))}
              </motion.ul>
            </div>
            <motion.div
              className="relative mx-auto w-full max-w-70 overflow-hidden rounded-xl ring-1 ring-slate-700/50 lg:max-w-none lg:col-span-5"
              initial={{ opacity: 0, y: 24 }}
              whileInView={{ opacity: 1, y: 0 }}
              viewport={{ once: true, amount: 0.2 }}
              transition={{ duration: 0.6, ease: [0.22, 1, 0.36, 1] }}
            >
              <div className="relative aspect-3/4 w-full">
                <Image
                  src="/images/why-us.jpg"
                  alt={t("imageAlt")}
                  fill
                  sizes="(max-width: 1024px) 280px, 33vw"
                  className="object-cover"
                />
              </div>
            </motion.div>
          </div>

          <motion.div
            className="mx-auto mt-20 grid max-w-4xl grid-cols-1 gap-8 md:grid-cols-2 text-left"
            initial="hidden"
            whileInView="visible"
            viewport={{ once: true, amount: 0.2 }}
            transition={{ staggerChildren: 0.15 }}
          >
            <PrincipleCard
              title={heroT("principles.collaboration.title") || "Collaboration"}
              description={heroT("principles.collaboration.description") || "Building strong partnerships."}
              icon={Handshake}
            />
            <PrincipleCard
              title={heroT("principles.sustainability.title") || "Sustainability"}
              description={heroT("principles.sustainability.description") || "Creating lasting impact."}
              icon={Leaf}
            />
          </motion.div>
        </div>
      </section>
    </>
  );
}
